<?php
/**
 * Helper functions used in the WP Post Modules plugin
 */

// Return parameters for BFI image resize
if ( ! function_exists( 'wppm_image_resize' ) ) :
	function wppm_image_resize( $src, $imgwidth, $imgheight, $imgcrop, $imgquality, $imgcolor, $imggrayscale ) {
		$params = array();

		// Validate boolean params
		$crop = ( '' == $imgcrop || 'false' == $imgcrop ) ? false : true;
		$grayscale = ( '' == $imggrayscale || 'false' == $imggrayscale ) ? false : true;

		// Params array
		if ( $crop ) {
			$params['crop'] = true;
		}

		if ( $grayscale ) {
			$params['grayscale'] = true;
		}

		if ( '' != $imgquality ) {
			if ( (int)$imgquality < 1 ) {
				$quality = 1;
			} elseif ( (int)$imgquality > 100 ) {
				$quality = 100;
			} else {
				$quality = $imgquality;
			}
			$params['quality'] = (int)$quality;
		}

		if ( '' != $imgcolor ) {
			$color = preg_match( '/#([a-f]|[A-F]|[0-9]){3}(([a-f]|[A-F]|[0-9]){3})?\b/', $imgcolor ) ? $imgcolor : '';
			$params['color'] = $color;
		}

		// Validate width and height
		if ( isset( $imgwidth ) && (int)$imgwidth > 4 && '' != $imgwidth ) {
			$params['width'] = $imgwidth;
		}

		if ( isset( $imgheight ) && (int)$imgheight > 4 && '' != $imgheight ) {
			$params['height'] = $imgheight;
		}

		if ( function_exists( 'bfi_thumb' ) && ! empty( $params ) ) {
			return bfi_thumb( $src, $params );
		} else {
			return $src;
		}
	}
endif;

// Remove p and br tags from nested short codes
if ( ! function_exists( 'wppm_clean' ) ) :
	function wppm_clean( $content, $p_tag = false, $br_tag = false ) {
		$content = preg_replace( '#^<\/p>|^<br \/>|<p>$#', '', $content );

		if ( $br_tag )
			$content = preg_replace( '#<br \/>#', '', $content );

		if ( $p_tag )
			$content = preg_replace( '#<p>|</p>#', '', $content );

		$array = array (
			'<p>[' => '[',
			']</p>' => ']',
			']<br />' => ']',
			']<br>' => ']',
			'<p></p>' => ''
		);

		$content = strtr( $content, $array );

		return apply_filters( 'the_content', do_shortcode( shortcode_unautop( trim( $content ) ) ) );
	}
endif;


/**
 * Function to shorten any text by word
 */
if ( ! function_exists( 'wppm_short' ) ) :
	function wppm_short( $phrase, $max_words = 10, $allowed_tags = 'p,br,a,em,i,strong,b' ) {

		$more = apply_filters( 'wppm_more_text', '&hellip;' );

		/**
		 * For Chinese and Japanese languages
		 * Trimming of multi byte text is different
		 */
		if ( in_array( get_locale(), apply_filters( 'wppm_multi_byte_lang_list', array( 'zh_CN', 'zh_HK', 'zh_TW', 'ja' ) ) ) ) {
			return wp_trim_words( $phrase, $max_words, $more );
		}

		else {
			if ( '' == $max_words ) {
				$max_words = 100;
			}
			$phrase_array = explode( ' ', $phrase );
			if ( count( $phrase_array ) > $max_words && $max_words > 0 ) {
				$phrase = implode( ' ', array_slice( $phrase_array, 0, $max_words ) );
			}
			$phrase = do_shortcode( $phrase );
			return strip_tags( $phrase, $allowed_tags ) . ' ' . $more;
		}
	}
endif;

/**
 * Parse attributes of a nested shortcode
 * https://wordpress.org/plugins/bootstrap-3-shortcodes/
 */

function wppm_parse_nested_atts($str, $att = null) {
	$result = array();
	$return = array();
	$reg = get_shortcode_regex();
	preg_match_all('~'.$reg.'~',$str, $matches);
	foreach( $matches[2] as $key => $name ) {
		$parsed = shortcode_parse_atts($matches[3][$key]);
		$parsed = is_array($parsed) ? $parsed : array();
		$result[$name] = $parsed;
		$return[] = $result;
	}
	return $return;
}

// Post meta for post modules
if ( ! function_exists( 'wppm_meta' ) ) :
	function wppm_meta( $args = array() ) {
		global $post;
		$custom_link = get_post_meta( $post->ID, 'wppm_custom_link', true );
			$defaults = array(
				'template'	=> 'grid',
				'date_format' => get_option( 'date_format' ),
				'enable_schema' => false,
				'hide_cats' => false,
				'hide_reviews' => false,
				'show_cats' => false,
				'show_reviews' => false,
				'hide_date' => false,
				'hide_author' => false,
				'show_avatar' => false,
				'hide_views' => false,
				'hide_comments' => false,
				'readmore' => false,
				'ext_link' => false,
				'readmore_text' => esc_attr__( 'Read more', 'wppm' ),
				'readmore_link_src' 	=> 'permalink',
				'readmore_target_blank' => false,
				'readmore_cust_link' 	=> '',
				'sharing'	=> false,
				'share_style'	=> 'popup',
				'share_btns' => '',

				// Schema props
				'datecreated_prop'		=> 'datePublished',
				'datemodified_prop'		=> 'dateModified',
				'publisher_type'		=> 'Organization',
				'publisher_prop'		=> 'publisher',
				'publisher_name'		=> esc_attr( get_bloginfo( 'name' ) ),
				'publisher_logo'		=> '',
				'authorbox_type'		=> 'Person',
				'authorbox_prop'		=> 'author',
				'authorname_prop'		=> 'name',
				'authoravatar_prop'		=> 'image',
				'category_prop'			=> 'about',
				'commentcount_prop'		=> 'commentCount',
				'commenturl_prop'		=> 'discussionUrl',
			);

			$args = wp_parse_args( $args, $defaults );

			extract( $args );

			$protocol = is_ssl() ? 'https' : 'http';
			$schema = $protocol . '://schema.org/';
			// Date format
			$date = get_the_time( get_option( 'date_format' ) );

			if ( ! empty( $date_format ) ) {
				if ( $date_format == 'human' ) {
					$date = sprintf( _x( '%s ago', 'human time difference. E.g. 10 days ago', 'wppm' ), human_time_diff( get_the_time( 'U' ), current_time( 'timestamp' ) ) );
				}
				else {
					$date = get_the_time( esc_attr( $date_format ) );
				}
			}

			// Category and review stars
			$review_meta = '';

			// Create category list
			$cat_list = '';
			$cat_out = '';
			$hasmore = false;
			$i = 0;
			$cats = get_the_category();
			$cat_limit = apply_filters( 'wppm_cat_list_limit', 3 );
			$cat_count = intval( count( $cats ) - $cat_limit );
			if ( isset( $cats ) ) {
				foreach( $cats as $cat ) {
					if ( $i == $cat_limit ) {
						$hasmore = true;
						$cat_list .= '<li class="submenu-parent"><a class="wppm-cat-toggle" href="#">' . sprintf( esc_attr_x( '+ %d more', 'more count for category list', 'wppm' ), number_format_i18n( $cat_count ) ) . '</a><ul class="cat-sub submenu">';
					}
					$cat_list .= '<li><a class="cat-' . $cat->slug . '" href="' . get_category_link( $cat->cat_ID ) . '">' . $cat->cat_name . '</a></li>';
					$i++;
				}
				if ( $cat_list ) {
					$cat_out = '<ul class="post-cats">' . $cat_list;
					$cat_out .= $hasmore ? '</ul></li></ul>' : '</ul>';
				}
			}

			$cat_meta = ( 'true' != $hide_cats ) ? $cat_out : '';
			if ( function_exists( 'wp_review_show_total' ) && 'true' !== $hide_reviews ) {
				$review_meta = wp_review_show_total( $echo = false );
			}

			// Author and date meta
			$meta_data = '';

			$author = get_the_author();
			if ( $show_avatar ) {
				$meta_data .= sprintf( '<div%s%s class="author-avatar-32%s"><a%s href="%s" title="%s">%s%s</a></div>',
					$enable_schema ? ' itemscope itemtype="' . $schema . $authorbox_type . '"' : '',
					$enable_schema ? ' itemprop="' . $authorbox_prop . '"' : '',
					$hide_author && $hide_date ? ' avatar-only' : '',
					$enable_schema ? ' itemprop="' . $authorname_prop . '"' : '',
					esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
					sprintf( esc_html__( 'More posts by %s', 'wppm' ), esc_attr( $author ) ),
					$enable_schema ? '<span itemprop="' . $authoravatar_prop . '">' . get_avatar( get_the_author_meta( 'user_email' ), 32 ) . '</span>' : get_avatar( get_the_author_meta( 'user_email' ), 32 ),
					$enable_schema ? '<span class="schema-only" itemprop="' . $authorname_prop . '">' . esc_attr( $author ) . '</span>' : ''

				);
			}

			$meta_data .= sprintf( '<ul class="entry-meta%s">',
				$show_avatar ? ' avatar-enabled' : ''
			);

			// Publisher Schema
			if ( $enable_schema ) {
				$meta_data .= '<li class="publisher-schema" itemscope itemtype="' . $schema . $publisher_type . '" itemprop="' . $publisher_prop . '"><meta itemprop="name" content="' . $publisher_name . '"/><div itemprop="logo" itemscope itemtype="' . $schema . 'ImageObject"><img itemprop="url" src="' . esc_url( $publisher_logo ) . '" alt="' . esc_attr( get_bloginfo( 'name' ) ) . '"/></div></li>';
			}

			$modified_date_format = 'human' == $date_format ? get_option( 'date_format' ) : $date_format;
			$meta_data .= sprintf( '<li class="post-time%1$s"><time%2$s class="entry-date updated" datetime="%3$s">%4$s</time>%5$s</li>',
				$hide_date ? ' schema-only' : '',
				$enable_schema ? ' itemprop="' . $datecreated_prop . '"' : '',
				esc_attr( get_the_date( 'c' ) ),
				esc_html( $date ),
				$enable_schema ? '<meta itemprop="' . $datemodified_prop . '" content="' . esc_attr( the_modified_date( $modified_date_format, '', '', false ) ) . '"/>' : ''
			);

			$meta_data .= sprintf( '<li%1$s%2$s class="post-author%3$s"><span class="screen-reader-text">%4$s</span><a href="%5$s">%6$s</a></li>',
				$enable_schema ? ' itemscope itemtype="' . $schema . $authorbox_type . '"' : '',
				$enable_schema ? ' itemprop="' . $authorbox_prop . '"' : '',
				$hide_author ? ' schema-only' : '',
				esc_html_x( 'Author', 'Used before post author name.', 'wppm' ),
				esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
				$enable_schema ? '<span class="vcard author ' . esc_attr( $author ) . '" itemprop="' . $authorname_prop . '"><span class="fn">' . esc_attr( $author ) . '</span></span>' : '<span class="vcard author ' . esc_attr( $author ) . '"><span class="fn">' . esc_attr( $author ) . '</span></span>'
			);

			$meta_data .= '</ul>';

			// Comment link
			$num_comments = get_comments_number();
			$comment_meta = '';
			if ( comments_open() && ( $num_comments >= 1 ) && ! $hide_comments ) {
				$comment_meta = sprintf( '<a href="%1$s" class="post-comment" title="%2$s">%3$s%4$s</a>',
					esc_url( get_comments_link() ),
					sprintf( __( 'Comment on %s', 'wppm' ), esc_attr( get_the_title() ) ),
					$enable_schema ? '<meta itemprop="' . $commenturl_prop . '" content="' . esc_url( get_comments_link() ) . '" />' : '',
					$enable_schema ? '<span itemprop="' . $commentcount_prop . '">' . $num_comments . '</span>' : $num_comments
				);
			}

			/**
			 * Post views
			 * Requires Plugin https://wordpress.org/plugins/post-views-counter/
			 */
			$views_meta = '';
			if ( function_exists( 'pvc_get_post_views' ) && ! $hide_views ) {
				$views_meta = sprintf( '<span class="post-views">%s</span>',
					pvc_get_post_views()
				);
			}

			/**
			 * Social share buttons
			 * Uses wppm_share_btns() function
			 */
			$share_btns_output =  $sharing ? wppm_share_btns( $share_btns, $share_style ) : '';

			// Generate rows of content
			$row_1 = '';
			$row_2 = '';
			$row_3 = '';
			$row_4 = '';
			if ( $review_meta != '' || $cat_meta != '' ) {
				$row_1 .= '<aside class="meta-row cat-row">';
				if ( $cat_meta != '' ) {
					$row_1 .= sprintf( '<div%s class="meta-col%s">%s</div>',
						$enable_schema ? ' itemprop="' . $category_prop . '"' : '',
						$review_meta != '' ? ' col-60' : '',
						$cat_meta
					);
				}

				if ( $review_meta != '' ) {
					$row_1 .= sprintf( '<div class="meta-col%s">%s</div>',
						$cat_meta != '' ? ' col-40 text-right' : '',
						$review_meta
					);
				}
				$row_1 .= '</aside>';
			}

			$row_4 .= sprintf( '<aside class="meta-row row-3%s">',
				( $hide_date && $hide_author && $hide_views && $hide_comments && 'true' !== $sharing ) ? ' schema-only' : ''
			);

			if ( '' == $views_meta && '' == $comment_meta && 'true' !== $sharing ) {
				$row_4 .= sprintf( '<div class="meta-col">%s</div>', $meta_data );
			}

			elseif ( '' == $meta_data ) {
				$row_4 .= sprintf( '<div class="meta-col">%s%s%s</div>', $views_meta, $comment_meta, $share_btns_output );
			}

			elseif ( 'inline' == $share_style ) {
				$row_4 .= sprintf( '<div class="meta-col col-60">%s</div><div class="meta-col col-40 text-right">%s%s</div>%s', $meta_data, $views_meta, $comment_meta, $share_btns_output );
			}

			else {
				$row_4 .= sprintf( '<div class="meta-col col-60">%s</div><div class="meta-col col-40 text-right">%s%s%s</div>', $meta_data, $views_meta, $comment_meta, $share_btns_output );
			}
			$row_4 .= '</aside>';

			if ( $readmore ) {
				if ( $meta_data ) {
					$row_2 = sprintf( '<aside class="meta-row row-2%s"><div class="meta-col">%s</div></aside>',
						( $hide_date && $hide_author && $hide_views && $hide_comments && 'true' !== $sharing ) ? ' schema-only' : '',
						$meta_data
					);
				}

				if ( $readmore || $views_meta || $comment_meta || $sharing ) {
					$row_3 = sprintf( '<aside class="meta-row row-3"><div class="meta-col col-50"><a class="readmore-link" href="%s"%s>%s</a></div><div class="meta-col col-50 text-right">%s%s%s</div></aside>',
						'ext' == $readmore_link_src && '' !== $custom_link ? esc_url( $custom_link ) : ( 'custom' == $readmore_link_src && '' !== $readmore_cust_link ? esc_url( $readmore_cust_link ) : esc_url( get_permalink() ) ),
						$readmore_target_blank ? ' target="_blank"' : '',
						esc_attr( $readmore_text ),
						$views_meta,
						$comment_meta,
						$share_btns_output
					);
				}
			}

			else {
				$row_3 = $row_4;
			}

		$meta_arr = array();
		$meta_arr['row_1'] = $row_1;
		$meta_arr['row_2'] = $row_2;
		$meta_arr['row_3'] = $row_3;
		$meta_arr['row_4'] = $row_4;
		return $meta_arr;
	}
endif;

/**
 * Get color pallete from image
 * http://stackoverflow.com/questions/10290259/detect-main-colors-in-an-image-with-php#answer-41044459
 */
if ( ! function_exists( 'wppm_get_color_pallet' ) ) :
	function wppm_get_color_pallet( $imageURL, $palletSize = array( 16, 8 ) ) {

		if ( ! $imageURL ) {
			return false;
		}

		// Create pallet from jpeg image
		$img = imagecreatefromjpeg( $imageURL );

		// Scale down image
		$imgSizes = getimagesize( $imageURL );

		$resizedImg = imagecreatetruecolor( $palletSize[0], $palletSize[1] );

		imagecopyresized( $resizedImg, $img, 0, 0, 0, 0, $palletSize[0], $palletSize[1], $imgSizes[0], $imgSizes[1] );

		imagedestroy( $img );

		// Get collors in array
		$colors = array();

		for( $i = 0; $i < $palletSize[1]; $i++ ) {
			for( $j = 0; $j < $palletSize[0]; $j++ ) {
				$colors[] = dechex( imagecolorat( $resizedImg, $j, $i ) );
			}
		}

		imagedestroy( $resizedImg );

		// Remove duplicates
		$colors= array_unique( $colors );

		return $colors;

	}
endif;

/**
 * Get dominant color from a given pallete
 */
if ( ! function_exists( 'wppm_get_dominant_color' ) ) :
	function wppm_get_dominant_color( $pallet ) {

		$lsum = 0;
		$larr = array();

		if ( isset( $pallet ) && is_array( $pallet ) ) {
			foreach( $pallet as $key => $val ) {
				// Split hex value of color in RGB
				$r = hexdec( substr( $val, 0, 2 ) );
				$g = hexdec( substr( $val, 2, 2 ) );
				$b = hexdec( substr( $val, 4, 2 ) );

				// Calculate luma (brightness in an image)
				$luma = 0.2126 * $r + 0.7152 * $g + 0.0722 * $b;
				$larr[] = array( 'luma' => $luma, 'rgb' => $r . ',' . $g . ',' . $b, 'rgbsum' => intval($r + $g + $b) );
				$lsum = $lsum + $luma;
			}

			// Average luma from all available colors in pallet
			$lavg = $lsum / count( $pallet );
			$rgbsum = 0;

			/**
			 * Find dominant color
			 *
			 * Compares each luma value to luma average
			 * and returns the closest match
			 */
			$closest = null;
			foreach ( $larr as $item ) {
				if ( $closest === null || abs( (int)$lavg - (int)$closest ) > abs( (int)$item['luma'] - (int)$lavg ) ) {
					$closest = $item['rgb'];
					$dluma = $item['luma'];
				}
			}

			// Check if the pallet is too light and needs dark text
			$dark_text = $dluma > 178 ? 'true' : '';

			return array( 'rgb' => $closest, 'dark_text' => $dark_text );
		}

		else {
			return array( 'rgb' => '77, 77, 77', 'dark_text' => '' );
		}
	}
endif;

/**
 * Social Sharing feature on single posts
 */
if ( ! function_exists( 'wppm_share_btns' ) ) :
	function wppm_share_btns( $share_btns, $share_style = 'popup' ) {
		global $post;
		setup_postdata( $post );
		$share_btns = ( $share_btns ) ? explode( ',', $share_btns ) : '';

		// Set variables
		$out = '';
		$list = '';
		$share_image = '';
		$protocol = is_ssl() ? 'https' : 'http';

		if ( has_post_thumbnail( $post->ID ) ) {
			$share_image = wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) );
		}

		$share_content = strip_tags( get_the_excerpt() );

		if ( 'inline' == $share_style ) {
			$out .= '<ul class="wppm-sharing inline">';
		} else {
		$out .= sprintf( '<div class="wppm-sharing-container"><a class="share-trigger" title="%1$s"><span class="screen-reader-text">%1$s</span></a><ul class="wppm-sharing">', __( 'Share this post', 'wppm' ) );
		}

		if ( ! empty( $share_btns ) && is_array( $share_btns ) ) {
			foreach ( $share_btns as $button ) {

				switch( $button ) {

					case 'twitter':
						$list .= sprintf( '<li class="wppm-twitter"><a href="%s://twitter.com/home?status=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on twitter', 'wppm' ), esc_attr__( 'Twitter', 'wppm' ) );
					break;

					case 'facebook':
						$list .= sprintf( '<li class="wppm-facebook"><a href="%s://www.facebook.com/sharer/sharer.php?u=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on facebook', 'wppm' ), esc_attr__( 'Facebook', 'wppm' ) );
					break;

					case 'whatsapp':
						if ( wp_is_mobile() ) {
							$list .= sprintf( '<li class="wppm-whatsapp"><a href="whatsapp://send?text=%s" title="%s" data-action="share/whatsapp/share">%s</a></li>', urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on Whatsapp', 'wppm' ), esc_attr__( 'Whatsapp', 'wppm' ) );
						}
					break;

					case 'googleplus':
						$list .= sprintf( '<li class="wppm-googleplus"><a href="%s://plus.google.com/share?url=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on Google+', 'wppm' ), esc_attr__( 'Google+', 'wppm' ) );
					break;

					case 'linkedin':
						$list .= sprintf( '<li class="wppm-linkedin"><a href="%s://www.linkedin.com/shareArticle?mini=true&amp;url=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on LinkedIn', 'wppm' ), esc_attr__( 'LinkedIn', 'wppm' ) );
					break;

					case 'pinterest':
						$list .= sprintf( '<li class="wppm-pinterest"><a href="%s://pinterest.com/pin/create/button/?url=%s&amp;media=%s" target="_blank" title="%s">%s</a></li>',
							esc_attr( $protocol ),
							urlencode( esc_url( get_permalink() ) ),
							esc_url( $share_image ),
							esc_attr__( 'Pin it', 'wppm' ),
							esc_attr__( 'Pinterest', 'wppm' )
						);
					break;

					case 'vkontakte':
						$list .= sprintf( '<li class="wppm-vkontakte"><a href="%s://vkontakte.ru/share.php?url=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share via VK', 'wppm' ), esc_attr__( 'VKOntakte', 'wppm' ) );
					break;

					case 'line':
						$list .= sprintf( '<li class="wppm-line"><a href="%s://social-plugins.line.me/lineit/share?url=%s" target="_blank" title="%s">%s</a></li>', esc_attr( $protocol ), urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on LINE', 'wppm' ), esc_attr__( 'LINE', 'wppm' ) );
					break;

					case 'reddit':
						$list .= sprintf( '<li class="wppm-reddit"><a href="//www.reddit.com/submit?url=%s" title="%s">%s</a></li>', urlencode( esc_url( get_permalink() ) ), esc_attr__( 'Share on Reddit', 'wppm' ), esc_attr__( 'Reddit', 'wppm' ) );
					break;

					case 'email':
						$list .= sprintf( '<li class="wppm-email no-popup"><a href="mailto:someone@example.com?Subject=%s" title="%s">%s</a></li>', urlencode( esc_attr( get_the_title() ) ), esc_attr__( 'Email this', 'wppm' ), esc_attr__( 'Email', 'wppm' ) );

					break;
				} // switch

			} // foreach
		} // if

		// Support extra meta items via action hook
		ob_start();
		do_action( 'wppm_sharing_buttons_li' );
		$out .= ob_get_contents();
		ob_end_clean();

		if ( 'inline' == $share_style ) {
			$out .= $list . '</ul>';
		} else {
			$out .= $list . '</ul></div>';
		}
		return $out;
	}
endif;

/**
 * Add Social meta tags for Facebook and Google+
 */
function wppm_social_meta_tags() {

	if ( is_single() && apply_filters( 'wppm_enable_social_tags', 'true' ) ) {
		global $post;
		setup_postdata( $post );
		$image = '';
		if ( has_post_thumbnail( $post->ID ) ) {
			$image = wp_get_attachment_url( get_post_thumbnail_id( $post->ID ) );
		}
		$title = get_the_title();
		$permalink = get_permalink();
		$excerpt = get_the_excerpt();
		$site_name = get_bloginfo( 'name' );
		?>

        <!-- Begin Social Meta tags by WP Post Modules -->
            <!-- Schema.org markup for Google+ -->
            <meta itemprop="name" content="<?php echo esc_attr( $title ); ?>">
            <meta itemprop="description" content="<?php echo strip_tags( $excerpt ); ?>">
            <meta itemprop="image" content="<?php echo esc_url( $image ); ?>">

            <!-- Twitter Card data -->
            <meta name="twitter:card" content="summary_large_image">
            <meta name="twitter:title" content="<?php echo esc_attr( $title ); ?>">
            <meta name="twitter:description" content="<?php echo strip_tags( $excerpt ); ?>">
            <meta name="twitter:image:src" content="<?php echo esc_url( $image ); ?>">

            <!-- Open Graph data -->
            <meta property="og:title" content="<?php echo esc_attr( $title ); ?>" />
            <meta property="og:type" content="article" />
            <meta property="og:url" content="<?php echo esc_url( $permalink ); ?>" />
            <meta property="og:image" content="<?php echo esc_url( $image ); ?>" />
            <meta property="og:image:url" content="<?php echo esc_url( $image ); ?>" />
            <meta property="og:description" content="<?php echo strip_tags( $excerpt ); ?>" />
            <meta property="og:site_name" content="<?php echo esc_attr( $site_name ); ?>" />
        <!-- End Social Meta tags by WP Post Modules -->
		<?php wp_reset_postdata();
	}
}
add_action( 'wp_head', 'wppm_social_meta_tags', 99 );

function wppm_animate_in_list() {
	return apply_filters( 'wppm_animate_in_list', array(

				__( 'Default', 'wppm' ) =>	array(
						__( 'Slide', 'wppm' ) => '',
					),

				__( 'Fading Entrances', 'wppm' )	=>	array(
					"fadeIn",
					"fadeInDown",
					"fadeInDownBig",
					"fadeInLeft",
					"fadeInLeftBig",
					"fadeInRight",
					"fadeInRightBig",
					"fadeInUp",
					"fadeInUpBig"
				),

				__( 'Sliding Entrances', 'wppm' )	=>	array(
					"slideInUp",
					"slideInDown",
					"slideInLeft",
					"slideInRight"
				),

				__( 'Zoom Entrances', 'wppm' )	=>	array(
					"zoomIn",
					"zoomInDown",
					"zoomInLeft",
					"zoomInRight",
					"zoomInUp"
				)
			)
		);
}

function wppm_animate_out_list() {
	return apply_filters( 'wppm_animate_out_list', array(

				__( 'Default', 'wppm' ) =>	array(
						__( 'Slide', 'wppm' ) => '',
					),

				__( 'Fading Exits', 'wppm' )	=>	array(
					"fadeOut",
					"fadeOutDown",
					"fadeOutDownBig",
					"fadeOutLeft",
					"fadeOutLeftBig",
					"fadeOutRight",
					"fadeOutRightBig",
					"fadeOutUp",
					"fadeOutUpBig"
				),

				__( 'Sliding Exits', 'wppm' )	=>	array(
					"slideOutUp",
					"slideOutDown",
					"slideOutLeft",
					"slideOutRight"
				),

				__( 'Zoom Exits', 'wppm' )	=>	array(
					"zoomOut",
					"zoomOutDown",
					"zoomOutLeft",
					"zoomOutRight",
					"zoomOutUp"
				)
		) );
}

if ( ! function_exists( 'wppm_enqueue_fonts' ) ) :
	function wppm_enqueue_fonts( $fontsData ) {
		// Get extra subsets for settings (latin/cyrillic/etc)
		$settings = get_option( 'wpb_js_google_fonts_subsets' );
		if ( is_array( $settings ) && ! empty( $settings ) ) {
			$subsets = '&subset=' . implode( ',', $settings );
		} else {
			$subsets = '';
		}

		// Enqueue font
		if ( isset( $fontsData ) ) {
			wp_enqueue_style( 'wppm_google_fonts_' . wppm_create_safe_class( $fontsData ), '//fonts.googleapis.com/css?family=' . $fontsData . $subsets );
		}
	}
endif;

if ( ! function_exists( 'wppm_create_font_style' ) ) :
	function wppm_create_font_style( $fontsData ) {
        $inline_style = '';
		if ( isset( $fontsData ) ) {
			$fontFamily = explode( ':', $fontsData );
			$styles[] = isset( $fontFamily[0] ) ? 'font-family:\'' . $fontFamily[0] . '\'': '';

			if ( isset( $styles ) && is_array( $styles ) ) {
				foreach( $styles as $attribute ){
					$inline_style .= $attribute.'; ';
				}
			}
		}
        return $inline_style;
    }
endif;

if ( ! function_exists( 'wppm_create_safe_class' ) ) :
	function wppm_create_safe_class( $class ) {
		return preg_replace( '/\W+/', '', strtolower( str_replace( ' ', '_', strip_tags( $class ) ) ) );
	}
endif;

/**
 * Get size information for all currently-registered image sizes.
 *
 * @global $_wp_additional_image_sizes
 * @uses   get_intermediate_image_sizes()
 * @return array $sizes Data for all currently-registered image sizes.
 */
if ( ! function_exists( 'wppm_get_image_sizes' ) ) :
function wppm_get_image_sizes() {
	global $_wp_additional_image_sizes;

	$sizes = array( __( 'Original', 'wppm' ) => 'full' );

	foreach ( get_intermediate_image_sizes() as $_size ) {
		if ( in_array( $_size, array('thumbnail', 'medium', 'medium_large', 'large') ) ) {
			/*$sizes[ $_size ]['width']  = get_option( "{$_size}_size_w" );
			$sizes[ $_size ]['height'] = get_option( "{$_size}_size_h" );
			$sizes[ $_size ]['crop']   = (bool) get_option( "{$_size}_crop" );*/
			$sizes[ $_size . '-' . get_option( "{$_size}_size_w" ) . 'x' . get_option( "{$_size}_size_h" ) ] = $_size;
		} elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {
			$sizes[ $_size . '-' . $_wp_additional_image_sizes[ $_size ]['width'] . 'x' . $_wp_additional_image_sizes[ $_size ]['height'] ] = $_size;
		}
	}

	return $sizes;
}
endif;

if ( ! function_exists( 'wppm_calc_image_sizes' ) ) :
function wppm_calc_image_sizes() {
	global $_wp_additional_image_sizes;

	foreach ( get_intermediate_image_sizes() as $_size ) {
		if ( in_array( $_size, array('thumbnail', 'medium', 'medium_large', 'large') ) ) {
			$sizes[ $_size ]['width']  = get_option( "{$_size}_size_w" );
			$sizes[ $_size ]['height'] = get_option( "{$_size}_size_h" );
		} elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {
			$sizes[ $_size ]['width']  = $_wp_additional_image_sizes[ $_size ]['width'];
			$sizes[ $_size ]['height'] = $_wp_additional_image_sizes[ $_size ]['width'];
		}
	}

	return $sizes;
}
endif;

/**
 * Get size information for a specific image size.
 *
 * @uses   wppm_get_image_sizes()
 * @param  string $size The image size for which to retrieve data.
 * @return bool|array $size Size data about an image size or false if the size doesn't exist.
 */
if ( ! function_exists( 'wppm_get_image_size' ) ) :
function wppm_get_image_size( $size ) {
	$sizes = wppm_calc_image_sizes();

	if ( isset( $sizes[ $size ] ) ) {
		return $sizes[ $size ];
	}

	return false;
}
endif;

/**
 * Get the width of a specific image size.
 *
 * @uses   wppm_get_image_size()
 * @param  string $size The image size for which to retrieve data.
 * @return bool|string $size Width of an image size or false if the size doesn't exist.
 */
if ( ! function_exists( 'wppm_get_image_width' ) ) :
function wppm_get_image_width( $size ) {
	if ( ! $size = wppm_get_image_size( $size ) ) {
		return false;
	}

	if ( isset( $size['width'] ) ) {
		return $size['width'];
	}

	return false;
}
endif;

/**
 * Get the height of a specific image size.
 *
 * @uses   wppm_get_image_size()
 * @param  string $size The image size for which to retrieve data.
 * @return bool|string $size Height of an image size or false if the size doesn't exist.
 */
if ( ! function_exists( 'wppm_get_image_height' ) ) :
	function wppm_get_image_height( $size ) {
		if ( ! $size = wppm_get_image_size( $size ) ) {
			return false;
		}

		if ( isset( $size['height'] ) ) {
			return $size['height'];
		}

		return false;
	}
endif;

if ( ! function_exists( 'wppm_google_font_list' ) ) :
function wppm_google_font_list() {
	return apply_filters( 'wppm_google_font_list',
				array(
					__( '-- Select --', 'wppm' ) => '',
					'ABeeZee' => 'ABeeZee:regular,italic',
					'Abel' => 'Abel:regular',
					'Abhaya Libre' => 'Abhaya Libre:regular,500,600,700,800',
					'Abril Fatface' => 'Abril Fatface:regular',
					'Aclonica' => 'Aclonica:regular',
					'Acme' => 'Acme:regular',
					'Actor' => 'Actor:regular',
					'Adamina' => 'Adamina:regular',
					'Advent Pro' => 'Advent Pro:100,200,300,regular,500,600,700',
					'Aguafina Script' => 'Aguafina Script:regular',
					'Akronim' => 'Akronim:regular',
					'Aladin' => 'Aladin:regular',
					'Aldrich' => 'Aldrich:regular',
					'Alef' => 'Alef:regular,700',
					'Alegreya' => 'Alegreya:regular,italic,700,700italic,900,900italic',
					'Alegreya SC' => 'Alegreya SC:regular,italic,700,700italic,900,900italic',
					'Alegreya Sans' => 'Alegreya Sans:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic',
					'Alegreya Sans SC' => 'Alegreya Sans SC:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic',
					'Alex Brush' => 'Alex Brush:regular',
					'Alfa Slab One' => 'Alfa Slab One:regular',
					'Alice' => 'Alice:regular',
					'Alike' => 'Alike:regular',
					'Alike Angular' => 'Alike Angular:regular',
					'Allan' => 'Allan:regular,700',
					'Allerta' => 'Allerta:regular',
					'Allerta Stencil' => 'Allerta Stencil:regular',
					'Allura' => 'Allura:regular',
					'Almendra' => 'Almendra:regular,italic,700,700italic',
					'Almendra Display' => 'Almendra Display:regular',
					'Almendra SC' => 'Almendra SC:regular',
					'Amarante' => 'Amarante:regular',
					'Amaranth' => 'Amaranth:regular,italic,700,700italic',
					'Amatic SC' => 'Amatic SC:regular,700',
					'Amatica SC' => 'Amatica SC:regular,700',
					'Amethysta' => 'Amethysta:regular',
					'Amiko' => 'Amiko:regular,600,700',
					'Amiri' => 'Amiri:regular,italic,700,700italic',
					'Amita' => 'Amita:regular,700',
					'Anaheim' => 'Anaheim:regular',
					'Andada' => 'Andada:regular',
					'Andika' => 'Andika:regular',
					'Angkor' => 'Angkor:regular',
					'Annie Use Your Telescope' => 'Annie Use Your Telescope:regular',
					'Anonymous Pro' => 'Anonymous Pro:regular,italic,700,700italic',
					'Antic' => 'Antic:regular',
					'Antic Didone' => 'Antic Didone:regular',
					'Antic Slab' => 'Antic Slab:regular',
					'Anton' => 'Anton:regular',
					'Arapey' => 'Arapey:regular,italic',
					'Arbutus' => 'Arbutus:regular',
					'Arbutus Slab' => 'Arbutus Slab:regular',
					'Architects Daughter' => 'Architects Daughter:regular',
					'Archivo Black' => 'Archivo Black:regular',
					'Archivo Narrow' => 'Archivo Narrow:regular,italic,700,700italic',
					'Aref Ruqaa' => 'Aref Ruqaa:regular,700',
					'Arima Madurai' => 'Arima Madurai:100,200,300,regular,500,700,800,900',
					'Arimo' => 'Arimo:regular,italic,700,700italic',
					'Arizonia' => 'Arizonia:regular',
					'Armata' => 'Armata:regular',
					'Arsenal' => 'Arsenal:regular,italic,700,700italic',
					'Artifika' => 'Artifika:regular',
					'Arvo' => 'Arvo:regular,italic,700,700italic',
					'Arya' => 'Arya:regular,700',
					'Asap' => 'Asap:regular,italic,500,500italic,700,700italic',
					'Asar' => 'Asar:regular',
					'Asset' => 'Asset:regular',
					'Assistant' => 'Assistant:200,300,regular,600,700,800',
					'Astloch' => 'Astloch:regular,700',
					'Asul' => 'Asul:regular,700',
					'Athiti' => 'Athiti:200,300,regular,500,600,700',
					'Atma' => 'Atma:300,regular,500,600,700',
					'Atomic Age' => 'Atomic Age:regular',
					'Aubrey' => 'Aubrey:regular',
					'Audiowide' => 'Audiowide:regular',
					'Autour One' => 'Autour One:regular',
					'Average' => 'Average:regular',
					'Average Sans' => 'Average Sans:regular',
					'Averia Gruesa Libre' => 'Averia Gruesa Libre:regular',
					'Averia Libre' => 'Averia Libre:300,300italic,regular,italic,700,700italic',
					'Averia Sans Libre' => 'Averia Sans Libre:300,300italic,regular,italic,700,700italic',
					'Averia Serif Libre' => 'Averia Serif Libre:300,300italic,regular,italic,700,700italic',
					'Bad Script' => 'Bad Script:regular',
					'Bahiana' => 'Bahiana:regular',
					'Baloo' => 'Baloo:regular',
					'Baloo Bhai' => 'Baloo Bhai:regular',
					'Baloo Bhaina' => 'Baloo Bhaina:regular',
					'Baloo Chettan' => 'Baloo Chettan:regular',
					'Baloo Da' => 'Baloo Da:regular',
					'Baloo Paaji' => 'Baloo Paaji:regular',
					'Baloo Tamma' => 'Baloo Tamma:regular',
					'Baloo Thambi' => 'Baloo Thambi:regular',
					'Balthazar' => 'Balthazar:regular',
					'Bangers' => 'Bangers:regular',
					'Barrio' => 'Barrio:regular',
					'Basic' => 'Basic:regular',
					'Battambang' => 'Battambang:regular,700',
					'Baumans' => 'Baumans:regular',
					'Bayon' => 'Bayon:regular',
					'Belgrano' => 'Belgrano:regular',
					'Bellefair' => 'Bellefair:regular',
					'Belleza' => 'Belleza:regular',
					'BenchNine' => 'BenchNine:300,regular,700',
					'Bentham' => 'Bentham:regular',
					'Berkshire Swash' => 'Berkshire Swash:regular',
					'Bevan' => 'Bevan:regular',
					'Bigelow Rules' => 'Bigelow Rules:regular',
					'Bigshot One' => 'Bigshot One:regular',
					'Bilbo' => 'Bilbo:regular',
					'Bilbo Swash Caps' => 'Bilbo Swash Caps:regular',
					'BioRhyme' => 'BioRhyme:200,300,regular,700,800',
					'BioRhyme Expanded' => 'BioRhyme Expanded:200,300,regular,700,800',
					'Biryani' => 'Biryani:200,300,regular,600,700,800,900',
					'Bitter' => 'Bitter:regular,italic,700',
					'Black Ops One' => 'Black Ops One:regular',
					'Bokor' => 'Bokor:regular',
					'Bonbon' => 'Bonbon:regular',
					'Boogaloo' => 'Boogaloo:regular',
					'Bowlby One' => 'Bowlby One:regular',
					'Bowlby One SC' => 'Bowlby One SC:regular',
					'Brawler' => 'Brawler:regular',
					'Bree Serif' => 'Bree Serif:regular',
					'Bubblegum Sans' => 'Bubblegum Sans:regular',
					'Bubbler One' => 'Bubbler One:regular',
					'Buda' => 'Buda:300',
					'Buenard' => 'Buenard:regular,700',
					'Bungee' => 'Bungee:regular',
					'Bungee Hairline' => 'Bungee Hairline:regular',
					'Bungee Inline' => 'Bungee Inline:regular',
					'Bungee Outline' => 'Bungee Outline:regular',
					'Bungee Shade' => 'Bungee Shade:regular',
					'Butcherman' => 'Butcherman:regular',
					'Butterfly Kids' => 'Butterfly Kids:regular',
					'Cabin' => 'Cabin:regular,italic,500,500italic,600,600italic,700,700italic',
					'Cabin Condensed' => 'Cabin Condensed:regular,500,600,700',
					'Cabin Sketch' => 'Cabin Sketch:regular,700',
					'Caesar Dressing' => 'Caesar Dressing:regular',
					'Cagliostro' => 'Cagliostro:regular',
					'Cairo' => 'Cairo:200,300,regular,600,700,900',
					'Calligraffitti' => 'Calligraffitti:regular',
					'Cambay' => 'Cambay:regular,italic,700,700italic',
					'Cambo' => 'Cambo:regular',
					'Candal' => 'Candal:regular',
					'Cantarell' => 'Cantarell:regular,italic,700,700italic',
					'Cantata One' => 'Cantata One:regular',
					'Cantora One' => 'Cantora One:regular',
					'Capriola' => 'Capriola:regular',
					'Cardo' => 'Cardo:regular,italic,700',
					'Carme' => 'Carme:regular',
					'Carrois Gothic' => 'Carrois Gothic:regular',
					'Carrois Gothic SC' => 'Carrois Gothic SC:regular',
					'Carter One' => 'Carter One:regular',
					'Catamaran' => 'Catamaran:100,200,300,regular,500,600,700,800,900',
					'Caudex' => 'Caudex:regular,italic,700,700italic',
					'Caveat' => 'Caveat:regular,700',
					'Caveat Brush' => 'Caveat Brush:regular',
					'Cedarville Cursive' => 'Cedarville Cursive:regular',
					'Ceviche One' => 'Ceviche One:regular',
					'Changa' => 'Changa:200,300,regular,500,600,700,800',
					'Changa One' => 'Changa One:regular,italic',
					'Chango' => 'Chango:regular',
					'Chathura' => 'Chathura:100,300,regular,700,800',
					'Chau Philomene One' => 'Chau Philomene One:regular,italic',
					'Chela One' => 'Chela One:regular',
					'Chelsea Market' => 'Chelsea Market:regular',
					'Chenla' => 'Chenla:regular',
					'Cherry Cream Soda' => 'Cherry Cream Soda:regular',
					'Cherry Swash' => 'Cherry Swash:regular,700',
					'Chewy' => 'Chewy:regular',
					'Chicle' => 'Chicle:regular',
					'Chivo' => 'Chivo:300,300italic,regular,italic,700,700italic,900,900italic',
					'Chonburi' => 'Chonburi:regular',
					'Cinzel' => 'Cinzel:regular,700,900',
					'Cinzel Decorative' => 'Cinzel Decorative:regular,700,900',
					'Clicker Script' => 'Clicker Script:regular',
					'Coda' => 'Coda:regular,800',
					'Coda Caption' => 'Coda Caption:800',
					'Codystar' => 'Codystar:300,regular',
					'Coiny' => 'Coiny:regular',
					'Combo' => 'Combo:regular',
					'Comfortaa' => 'Comfortaa:300,regular,700',
					'Coming Soon' => 'Coming Soon:regular',
					'Concert One' => 'Concert One:regular',
					'Condiment' => 'Condiment:regular',
					'Content' => 'Content:regular,700',
					'Contrail One' => 'Contrail One:regular',
					'Convergence' => 'Convergence:regular',
					'Cookie' => 'Cookie:regular',
					'Copse' => 'Copse:regular',
					'Corben' => 'Corben:regular,700',
					'Cormorant' => 'Cormorant:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant Garamond' => 'Cormorant Garamond:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant Infant' => 'Cormorant Infant:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic',
					'Cormorant SC' => 'Cormorant SC:300,regular,500,600,700',
					'Cormorant Unicase' => 'Cormorant Unicase:300,regular,500,600,700',
					'Cormorant Upright' => 'Cormorant Upright:300,regular,500,600,700',
					'Courgette' => 'Courgette:regular',
					'Cousine' => 'Cousine:regular,italic,700,700italic',
					'Coustard' => 'Coustard:regular,900',
					'Covered By Your Grace' => 'Covered By Your Grace:regular',
					'Crafty Girls' => 'Crafty Girls:regular',
					'Creepster' => 'Creepster:regular',
					'Crete Round' => 'Crete Round:regular,italic',
					'Crimson Text' => 'Crimson Text:regular,italic,600,600italic,700,700italic',
					'Croissant One' => 'Croissant One:regular',
					'Crushed' => 'Crushed:regular',
					'Cuprum' => 'Cuprum:regular,italic,700,700italic',
					'Cutive' => 'Cutive:regular',
					'Cutive Mono' => 'Cutive Mono:regular',
					'Damion' => 'Damion:regular',
					'Dancing Script' => 'Dancing Script:regular,700',
					'Dangrek' => 'Dangrek:regular',
					'David Libre' => 'David Libre:regular,500,700',
					'Dawning of a New Day' => 'Dawning of a New Day:regular',
					'Days One' => 'Days One:regular',
					'Dekko' => 'Dekko:regular',
					'Delius' => 'Delius:regular',
					'Delius Swash Caps' => 'Delius Swash Caps:regular',
					'Delius Unicase' => 'Delius Unicase:regular,700',
					'Della Respira' => 'Della Respira:regular',
					'Denk One' => 'Denk One:regular',
					'Devonshire' => 'Devonshire:regular',
					'Dhurjati' => 'Dhurjati:regular',
					'Didact Gothic' => 'Didact Gothic:regular',
					'Diplomata' => 'Diplomata:regular',
					'Diplomata SC' => 'Diplomata SC:regular',
					'Domine' => 'Domine:regular,700',
					'Donegal One' => 'Donegal One:regular',
					'Doppio One' => 'Doppio One:regular',
					'Dorsa' => 'Dorsa:regular',
					'Dosis' => 'Dosis:200,300,regular,500,600,700,800',
					'Dr Sugiyama' => 'Dr Sugiyama:regular',
					'Droid Sans' => 'Droid Sans:regular,700',
					'Droid Sans Mono' => 'Droid Sans Mono:regular',
					'Droid Serif' => 'Droid Serif:regular,italic,700,700italic',
					'Duru Sans' => 'Duru Sans:regular',
					'Dynalight' => 'Dynalight:regular',
					'EB Garamond' => 'EB Garamond:regular',
					'Eagle Lake' => 'Eagle Lake:regular',
					'Eater' => 'Eater:regular',
					'Economica' => 'Economica:regular,italic,700,700italic',
					'Eczar' => 'Eczar:regular,500,600,700,800',
					'Ek Mukta' => 'Ek Mukta:200,300,regular,500,600,700,800',
					'El Messiri' => 'El Messiri:regular,500,600,700',
					'Electrolize' => 'Electrolize:regular',
					'Elsie' => 'Elsie:regular,900',
					'Elsie Swash Caps' => 'Elsie Swash Caps:regular,900',
					'Emblema One' => 'Emblema One:regular',
					'Emilys Candy' => 'Emilys Candy:regular',
					'Engagement' => 'Engagement:regular',
					'Englebert' => 'Englebert:regular',
					'Enriqueta' => 'Enriqueta:regular,700',
					'Erica One' => 'Erica One:regular',
					'Esteban' => 'Esteban:regular',
					'Euphoria Script' => 'Euphoria Script:regular',
					'Ewert' => 'Ewert:regular',
					'Exo' => 'Exo:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Exo 2' => 'Exo 2:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Expletus Sans' => 'Expletus Sans:regular,italic,500,500italic,600,600italic,700,700italic',
					'Fanwood Text' => 'Fanwood Text:regular,italic',
					'Farsan' => 'Farsan:regular',
					'Fascinate' => 'Fascinate:regular',
					'Fascinate Inline' => 'Fascinate Inline:regular',
					'Faster One' => 'Faster One:regular',
					'Fasthand' => 'Fasthand:regular',
					'Fauna One' => 'Fauna One:regular',
					'Federant' => 'Federant:regular',
					'Federo' => 'Federo:regular',
					'Felipa' => 'Felipa:regular',
					'Fenix' => 'Fenix:regular',
					'Finger Paint' => 'Finger Paint:regular',
					'Fira Mono' => 'Fira Mono:regular,500,700',
					'Fira Sans' => 'Fira Sans:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fira Sans Condensed' => 'Fira Sans Condensed:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fira Sans Extra Condensed' => 'Fira Sans Extra Condensed:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Fjalla One' => 'Fjalla One:regular',
					'Fjord One' => 'Fjord One:regular',
					'Flamenco' => 'Flamenco:300,regular',
					'Flavors' => 'Flavors:regular',
					'Fondamento' => 'Fondamento:regular,italic',
					'Fontdiner Swanky' => 'Fontdiner Swanky:regular',
					'Forum' => 'Forum:regular',
					'Francois One' => 'Francois One:regular',
					'Frank Ruhl Libre' => 'Frank Ruhl Libre:300,regular,500,700,900',
					'Freckle Face' => 'Freckle Face:regular',
					'Fredericka the Great' => 'Fredericka the Great:regular',
					'Fredoka One' => 'Fredoka One:regular',
					'Freehand' => 'Freehand:regular',
					'Fresca' => 'Fresca:regular',
					'Frijole' => 'Frijole:regular',
					'Fruktur' => 'Fruktur:regular',
					'Fugaz One' => 'Fugaz One:regular',
					'GFS Didot' => 'GFS Didot:regular',
					'GFS Neohellenic' => 'GFS Neohellenic:regular,italic,700,700italic',
					'Gabriela' => 'Gabriela:regular',
					'Gafata' => 'Gafata:regular',
					'Galada' => 'Galada:regular',
					'Galdeano' => 'Galdeano:regular',
					'Galindo' => 'Galindo:regular',
					'Gentium Basic' => 'Gentium Basic:regular,italic,700,700italic',
					'Gentium Book Basic' => 'Gentium Book Basic:regular,italic,700,700italic',
					'Geo' => 'Geo:regular,italic',
					'Geostar' => 'Geostar:regular',
					'Geostar Fill' => 'Geostar Fill:regular',
					'Germania One' => 'Germania One:regular',
					'Gidugu' => 'Gidugu:regular',
					'Gilda Display' => 'Gilda Display:regular',
					'Give You Glory' => 'Give You Glory:regular',
					'Glass Antiqua' => 'Glass Antiqua:regular',
					'Glegoo' => 'Glegoo:regular,700',
					'Gloria Hallelujah' => 'Gloria Hallelujah:regular',
					'Goblin One' => 'Goblin One:regular',
					'Gochi Hand' => 'Gochi Hand:regular',
					'Gorditas' => 'Gorditas:regular,700',
					'Goudy Bookletter 1911' => 'Goudy Bookletter 1911:regular',
					'Graduate' => 'Graduate:regular',
					'Grand Hotel' => 'Grand Hotel:regular',
					'Gravitas One' => 'Gravitas One:regular',
					'Great Vibes' => 'Great Vibes:regular',
					'Griffy' => 'Griffy:regular',
					'Gruppo' => 'Gruppo:regular',
					'Gudea' => 'Gudea:regular,italic,700',
					'Gurajada' => 'Gurajada:regular',
					'Habibi' => 'Habibi:regular',
					'Halant' => 'Halant:300,regular,500,600,700',
					'Hammersmith One' => 'Hammersmith One:regular',
					'Hanalei' => 'Hanalei:regular',
					'Hanalei Fill' => 'Hanalei Fill:regular',
					'Handlee' => 'Handlee:regular',
					'Hanuman' => 'Hanuman:regular,700',
					'Happy Monkey' => 'Happy Monkey:regular',
					'Harmattan' => 'Harmattan:regular',
					'Headland One' => 'Headland One:regular',
					'Heebo' => 'Heebo:100,300,regular,500,700,800,900',
					'Henny Penny' => 'Henny Penny:regular',
					'Herr Von Muellerhoff' => 'Herr Von Muellerhoff:regular',
					'Hind' => 'Hind:300,regular,500,600,700',
					'Hind Guntur' => 'Hind Guntur:300,regular,500,600,700',
					'Hind Madurai' => 'Hind Madurai:300,regular,500,600,700',
					'Hind Siliguri' => 'Hind Siliguri:300,regular,500,600,700',
					'Hind Vadodara' => 'Hind Vadodara:300,regular,500,600,700',
					'Holtwood One SC' => 'Holtwood One SC:regular',
					'Homemade Apple' => 'Homemade Apple:regular',
					'Homenaje' => 'Homenaje:regular',
					'IM Fell DW Pica' => 'IM Fell DW Pica:regular,italic',
					'IM Fell DW Pica SC' => 'IM Fell DW Pica SC:regular',
					'IM Fell Double Pica' => 'IM Fell Double Pica:regular,italic',
					'IM Fell Double Pica SC' => 'IM Fell Double Pica SC:regular',
					'IM Fell English' => 'IM Fell English:regular,italic',
					'IM Fell English SC' => 'IM Fell English SC:regular',
					'IM Fell French Canon' => 'IM Fell French Canon:regular,italic',
					'IM Fell French Canon SC' => 'IM Fell French Canon SC:regular',
					'IM Fell Great Primer' => 'IM Fell Great Primer:regular,italic',
					'IM Fell Great Primer SC' => 'IM Fell Great Primer SC:regular',
					'Iceberg' => 'Iceberg:regular',
					'Iceland' => 'Iceland:regular',
					'Imprima' => 'Imprima:regular',

					'Inconsolata' => 'Inconsolata:regular,700',
					'Inder' => 'Inder:regular',
					'Indie Flower' => 'Indie Flower:regular',
					'Inika' => 'Inika:regular,700',
					'Inknut Antiqua' => 'Inknut Antiqua:300,regular,500,600,700,800,900',
					'Irish Grover' => 'Irish Grover:regular',
					'Istok Web' => 'Istok Web:regular,italic,700,700italic',
					'Italiana' => 'Italiana:regular',
					'Italianno' => 'Italianno:regular',
					'Itim' => 'Itim:regular',
					'Jacques Francois' => 'Jacques Francois:regular',
					'Jacques Francois Shadow' => 'Jacques Francois Shadow:regular',
					'Jaldi' => 'Jaldi:regular,700',
					'Jim Nightshade' => 'Jim Nightshade:regular',
					'Jockey One' => 'Jockey One:regular',
					'Jolly Lodger' => 'Jolly Lodger:regular',
					'Jomhuria' => 'Jomhuria:regular',
					'Josefin Sans' => 'Josefin Sans:100,100italic,300,300italic,regular,italic,600,600italic,700,700italic',
					'Josefin Slab' => 'Josefin Slab:100,100italic,300,300italic,regular,italic,600,600italic,700,700italic',
					'Joti One' => 'Joti One:regular',
					'Judson' => 'Judson:regular,italic,700',
					'Julee' => 'Julee:regular',
					'Julius Sans One' => 'Julius Sans One:regular',
					'Junge' => 'Junge:regular',
					'Jura' => 'Jura:300,regular,500,600,700',
					'Just Another Hand' => 'Just Another Hand:regular',
					'Just Me Again Down Here' => 'Just Me Again Down Here:regular',
					'Kadwa' => 'Kadwa:regular,700',
					'Kalam' => 'Kalam:300,regular,700',
					'Kameron' => 'Kameron:regular,700',
					'Kanit' => 'Kanit:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Kantumruy' => 'Kantumruy:300,regular,700',
					'Karla' => 'Karla:regular,italic,700,700italic',
					'Karma' => 'Karma:300,regular,500,600,700',
					'Katibeh' => 'Katibeh:regular',
					'Kaushan Script' => 'Kaushan Script:regular',
					'Kavivanar' => 'Kavivanar:regular',
					'Kavoon' => 'Kavoon:regular',
					'Kdam Thmor' => 'Kdam Thmor:regular',
					'Keania One' => 'Keania One:regular',
					'Kelly Slab' => 'Kelly Slab:regular',
					'Kenia' => 'Kenia:regular',
					'Khand' => 'Khand:300,regular,500,600,700',
					'Khmer' => 'Khmer:regular',
					'Khula' => 'Khula:300,regular,600,700,800',
					'Kite One' => 'Kite One:regular',
					'Knewave' => 'Knewave:regular',
					'Kotta One' => 'Kotta One:regular',
					'Koulen' => 'Koulen:regular',
					'Kranky' => 'Kranky:regular',
					'Kreon' => 'Kreon:300,regular,700',
					'Kristi' => 'Kristi:regular',
					'Krona One' => 'Krona One:regular',
					'Kumar One' => 'Kumar One:regular',
					'Kumar One Outline' => 'Kumar One Outline:regular',
					'Kurale' => 'Kurale:regular',
					'La Belle Aurore' => 'La Belle Aurore:regular',
					'Laila' => 'Laila:300,regular,500,600,700',
					'Lakki Reddy' => 'Lakki Reddy:regular',
					'Lalezar' => 'Lalezar:regular',
					'Lancelot' => 'Lancelot:regular',
					'Lateef' => 'Lateef:regular',
					'Lato' => 'Lato:100,100italic,300,300italic,regular,italic,700,700italic,900,900italic',
					'League Script' => 'League Script:regular',
					'Leckerli One' => 'Leckerli One:regular',
					'Ledger' => 'Ledger:regular',
					'Lekton' => 'Lekton:regular,italic,700',
					'Lemon' => 'Lemon:regular',
					'Lemonada' => 'Lemonada:300,regular,600,700',
					'Libre Baskerville' => 'Libre Baskerville:regular,italic,700',
					'Libre Franklin' => 'Libre Franklin:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Life Savers' => 'Life Savers:regular,700',
					'Lilita One' => 'Lilita One:regular',
					'Lily Script One' => 'Lily Script One:regular',
					'Limelight' => 'Limelight:regular',
					'Linden Hill' => 'Linden Hill:regular,italic',
					'Lobster' => 'Lobster:regular',
					'Lobster Two' => 'Lobster Two:regular,italic,700,700italic',
					'Londrina Outline' => 'Londrina Outline:regular',
					'Londrina Shadow' => 'Londrina Shadow:regular',
					'Londrina Sketch' => 'Londrina Sketch:regular',
					'Londrina Solid' => 'Londrina Solid:regular',
					'Lora' => 'Lora:regular,italic,700,700italic',
					'Love Ya Like A Sister' => 'Love Ya Like A Sister:regular',
					'Loved by the King' => 'Loved by the King:regular',
					'Lovers Quarrel' => 'Lovers Quarrel:regular',
					'Luckiest Guy' => 'Luckiest Guy:regular',
					'Lusitana' => 'Lusitana:regular,700',
					'Lustria' => 'Lustria:regular',
					'Macondo' => 'Macondo:regular',
					'Macondo Swash Caps' => 'Macondo Swash Caps:regular',
					'Mada' => 'Mada:300,regular,500,900',
					'Magra' => 'Magra:regular,700',
					'Maiden Orange' => 'Maiden Orange:regular',
					'Maitree' => 'Maitree:200,300,regular,500,600,700',
					'Mako' => 'Mako:regular',
					'Mallanna' => 'Mallanna:regular',
					'Mandali' => 'Mandali:regular',
					'Marcellus' => 'Marcellus:regular',
					'Marcellus SC' => 'Marcellus SC:regular',
					'Marck Script' => 'Marck Script:regular',
					'Margarine' => 'Margarine:regular',
					'Marko One' => 'Marko One:regular',
					'Marmelad' => 'Marmelad:regular',
					'Martel' => 'Martel:200,300,regular,600,700,800,900',
					'Martel Sans' => 'Martel Sans:200,300,regular,600,700,800,900',
					'Marvel' => 'Marvel:regular,italic,700,700italic',
					'Mate' => 'Mate:regular,italic',
					'Mate SC' => 'Mate SC:regular',
					'Maven Pro' => 'Maven Pro:regular,500,700,900',
					'McLaren' => 'McLaren:regular',
					'Meddon' => 'Meddon:regular',
					'MedievalSharp' => 'MedievalSharp:regular',
					'Medula One' => 'Medula One:regular',
					'Meera Inimai' => 'Meera Inimai:regular',
					'Megrim' => 'Megrim:regular',
					'Meie Script' => 'Meie Script:regular',
					'Merienda' => 'Merienda:regular,700',
					'Merienda One' => 'Merienda One:regular',
					'Merriweather' => 'Merriweather:300,300italic,regular,italic,700,700italic,900,900italic',
					'Merriweather Sans' => 'Merriweather Sans:300,300italic,regular,italic,700,700italic,800,800italic',
					'Metal' => 'Metal:regular',
					'Metal Mania' => 'Metal Mania:regular',
					'Metamorphous' => 'Metamorphous:regular',
					'Metrophobic' => 'Metrophobic:regular',
					'Michroma' => 'Michroma:regular',
					'Milonga' => 'Milonga:regular',
					'Miltonian' => 'Miltonian:regular',
					'Miltonian Tattoo' => 'Miltonian Tattoo:regular',
					'Miniver' => 'Miniver:regular',
					'Miriam Libre' => 'Miriam Libre:regular,700',
					'Mirza' => 'Mirza:regular,500,600,700',
					'Miss Fajardose' => 'Miss Fajardose:regular',
					'Mitr' => 'Mitr:200,300,regular,500,600,700',
					'Modak' => 'Modak:regular',
					'Modern Antiqua' => 'Modern Antiqua:regular',
					'Mogra' => 'Mogra:regular',
					'Molengo' => 'Molengo:regular',
					'Molle' => 'Molle:italic',
					'Monda' => 'Monda:regular,700',
					'Monofett' => 'Monofett:regular',
					'Monoton' => 'Monoton:regular',
					'Monsieur La Doulaise' => 'Monsieur La Doulaise:regular',
					'Montaga' => 'Montaga:regular',
					'Montez' => 'Montez:regular',
					'Montserrat' => 'Montserrat:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Montserrat Alternates' => 'Montserrat Alternates:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Montserrat Subrayada' => 'Montserrat Subrayada:regular,700',
					'Moul' => 'Moul:regular',
					'Moulpali' => 'Moulpali:regular',
					'Mountains of Christmas' => 'Mountains of Christmas:regular,700',
					'Mouse Memoirs' => 'Mouse Memoirs:regular',
					'Mr Bedfort' => 'Mr Bedfort:regular',
					'Mr Dafoe' => 'Mr Dafoe:regular',
					'Mr De Haviland' => 'Mr De Haviland:regular',
					'Mrs Saint Delafield' => 'Mrs Saint Delafield:regular',
					'Mrs Sheppards' => 'Mrs Sheppards:regular',
					'Mukta Vaani' => 'Mukta Vaani:200,300,regular,500,600,700,800',
					'Muli' => 'Muli:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Mystery Quest' => 'Mystery Quest:regular',
					'NTR' => 'NTR:regular',
					'Neucha' => 'Neucha:regular',
					'Neuton' => 'Neuton:200,300,regular,italic,700,800',
					'New Rocker' => 'New Rocker:regular',
					'News Cycle' => 'News Cycle:regular,700',
					'Niconne' => 'Niconne:regular',
					'Nixie One' => 'Nixie One:regular',
					'Nobile' => 'Nobile:regular,italic,700,700italic',
					'Nokora' => 'Nokora:regular,700',
					'Norican' => 'Norican:regular',
					'Nosifer' => 'Nosifer:regular',
					'Nothing You Could Do' => 'Nothing You Could Do:regular',
					'Noticia Text' => 'Noticia Text:regular,italic,700,700italic',
					'Noto Sans' => 'Noto Sans:regular,italic,700,700italic',
					'Noto Serif' => 'Noto Serif:regular,italic,700,700italic',
					'Nova Cut' => 'Nova Cut:regular',
					'Nova Flat' => 'Nova Flat:regular',
					'Nova Mono' => 'Nova Mono:regular',
					'Nova Oval' => 'Nova Oval:regular',
					'Nova Round' => 'Nova Round:regular',
					'Nova Script' => 'Nova Script:regular',
					'Nova Slim' => 'Nova Slim:regular',
					'Nova Square' => 'Nova Square:regular',
					'Numans' => 'Numans:regular',
					'Nunito' => 'Nunito:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Nunito Sans' => 'Nunito Sans:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Odor Mean Chey' => 'Odor Mean Chey:regular',
					'Offside' => 'Offside:regular',
					'Old Standard TT' => 'Old Standard TT:regular,italic,700',
					'Oldenburg' => 'Oldenburg:regular',
					'Oleo Script' => 'Oleo Script:regular,700',
					'Oleo Script Swash Caps' => 'Oleo Script Swash Caps:regular,700',
					'Open Sans' => 'Open Sans:300,300italic,regular,italic,600,600italic,700,700italic,800,800italic',
					'Open Sans Condensed' => 'Open Sans Condensed:300,300italic,700',
					'Oranienbaum' => 'Oranienbaum:regular',
					'Orbitron' => 'Orbitron:regular,500,700,900',
					'Oregano' => 'Oregano:regular,italic',
					'Orienta' => 'Orienta:regular',
					'Original Surfer' => 'Original Surfer:regular',
					'Oswald' => 'Oswald:200,300,regular,500,600,700',
					'Over the Rainbow' => 'Over the Rainbow:regular',
					'Overlock' => 'Overlock:regular,italic,700,700italic,900,900italic',
					'Overlock SC' => 'Overlock SC:regular',
					'Overpass' => 'Overpass:100,100italic,200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Overpass Mono' => 'Overpass Mono:300,regular,600,700',
					'Ovo' => 'Ovo:regular',
					'Oxygen' => 'Oxygen:300,regular,700',
					'Oxygen Mono' => 'Oxygen Mono:regular',
					'PT Mono' => 'PT Mono:regular',
					'PT Sans' => 'PT Sans:regular,italic,700,700italic',
					'PT Sans Caption' => 'PT Sans Caption:regular,700',
					'PT Sans Narrow' => 'PT Sans Narrow:regular,700',
					'PT Serif' => 'PT Serif:regular,italic,700,700italic',
					'PT Serif Caption' => 'PT Serif Caption:regular,italic',
					'Pacifico' => 'Pacifico:regular',
					'Padauk' => 'Padauk:regular,700',
					'Palanquin' => 'Palanquin:100,200,300,regular,500,600,700',
					'Palanquin Dark' => 'Palanquin Dark:regular,500,600,700',
					'Pangolin' => 'Pangolin:regular',
					'Paprika' => 'Paprika:regular',
					'Parisienne' => 'Parisienne:regular',
					'Passero One' => 'Passero One:regular',
					'Passion One' => 'Passion One:regular,700,900',
					'Pathway Gothic One' => 'Pathway Gothic One:regular',
					'Patrick Hand' => 'Patrick Hand:regular',
					'Patrick Hand SC' => 'Patrick Hand SC:regular',
					'Pattaya' => 'Pattaya:regular',
					'Patua One' => 'Patua One:regular',
					'Pavanam' => 'Pavanam:regular',
					'Paytone One' => 'Paytone One:regular',
					'Peddana' => 'Peddana:regular',
					'Peralta' => 'Peralta:regular',
					'Permanent Marker' => 'Permanent Marker:regular',
					'Petit Formal Script' => 'Petit Formal Script:regular',
					'Petrona' => 'Petrona:regular',
					'Philosopher' => 'Philosopher:regular,italic,700,700italic',
					'Piedra' => 'Piedra:regular',
					'Pinyon Script' => 'Pinyon Script:regular',
					'Pirata One' => 'Pirata One:regular',
					'Plaster' => 'Plaster:regular',
					'Play' => 'Play:regular,700',
					'Playball' => 'Playball:regular',
					'Playfair Display' => 'Playfair Display:regular,italic,700,700italic,900,900italic',
					'Playfair Display SC' => 'Playfair Display SC:regular,italic,700,700italic,900,900italic',
					'Podkova' => 'Podkova:regular,500,600,700,800',
					'Poiret One' => 'Poiret One:regular',
					'Poller One' => 'Poller One:regular',
					'Poly' => 'Poly:regular,italic',
					'Pompiere' => 'Pompiere:regular',
					'Pontano Sans' => 'Pontano Sans:regular',
					'Poppins' => 'Poppins:300,regular,500,600,700',
					'Port Lligat Sans' => 'Port Lligat Sans:regular',
					'Port Lligat Slab' => 'Port Lligat Slab:regular',
					'Pragati Narrow' => 'Pragati Narrow:regular,700',
					'Prata' => 'Prata:regular',
					'Preahvihear' => 'Preahvihear:regular',
					'Press Start 2P' => 'Press Start 2P:regular',
					'Pridi' => 'Pridi:200,300,regular,500,600,700',
					'Princess Sofia' => 'Princess Sofia:regular',
					'Prociono' => 'Prociono:regular',
					'Prompt' => 'Prompt:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Prosto One' => 'Prosto One:regular',
					'Proza Libre' => 'Proza Libre:regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Puritan' => 'Puritan:regular,italic,700,700italic',
					'Purple Purse' => 'Purple Purse:regular',
					'Quando' => 'Quando:regular',
					'Quantico' => 'Quantico:regular,italic,700,700italic',
					'Quattrocento' => 'Quattrocento:regular,700',
					'Quattrocento Sans' => 'Quattrocento Sans:regular,italic,700,700italic',
					'Questrial' => 'Questrial:regular',
					'Quicksand' => 'Quicksand:300,regular,500,700',
					'Quintessential' => 'Quintessential:regular',
					'Qwigley' => 'Qwigley:regular',
					'Racing Sans One' => 'Racing Sans One:regular',
					'Radley' => 'Radley:regular,italic',
					'Rajdhani' => 'Rajdhani:300,regular,500,600,700',
					'Rakkas' => 'Rakkas:regular',
					'Raleway' => 'Raleway:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Raleway Dots' => 'Raleway Dots:regular',
					'Ramabhadra' => 'Ramabhadra:regular',
					'Ramaraja' => 'Ramaraja:regular',
					'Rambla' => 'Rambla:regular,italic,700,700italic',
					'Rammetto One' => 'Rammetto One:regular',
					'Ranchers' => 'Ranchers:regular',
					'Rancho' => 'Rancho:regular',
					'Ranga' => 'Ranga:regular,700',
					'Rasa' => 'Rasa:300,regular,500,600,700',
					'Rationale' => 'Rationale:regular',
					'Ravi Prakash' => 'Ravi Prakash:regular',
					'Redressed' => 'Redressed:regular',
					'Reem Kufi' => 'Reem Kufi:regular',
					'Reenie Beanie' => 'Reenie Beanie:regular',
					'Revalia' => 'Revalia:regular',
					'Rhodium Libre' => 'Rhodium Libre:regular',
					'Ribeye' => 'Ribeye:regular',
					'Ribeye Marrow' => 'Ribeye Marrow:regular',
					'Righteous' => 'Righteous:regular',
					'Risque' => 'Risque:regular',
					'Roboto' => 'Roboto:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,900,900italic',
					'Roboto Condensed' => 'Roboto Condensed:300,300italic,regular,italic,700,700italic',
					'Roboto Mono' => 'Roboto Mono:100,100italic,300,300italic,regular,italic,500,500italic,700,700italic',
					'Roboto Slab' => 'Roboto Slab:100,300,regular,700',
					'Rochester' => 'Rochester:regular',
					'Rock Salt' => 'Rock Salt:regular',
					'Rokkitt' => 'Rokkitt:100,200,300,regular,500,600,700,800,900',
					'Romanesco' => 'Romanesco:regular',
					'Ropa Sans' => 'Ropa Sans:regular,italic',
					'Rosario' => 'Rosario:regular,italic,700,700italic',
					'Rosarivo' => 'Rosarivo:regular,italic',
					'Rouge Script' => 'Rouge Script:regular',
					'Rozha One' => 'Rozha One:regular',
					'Rubik' => 'Rubik:300,300italic,regular,italic,500,500italic,700,700italic,900,900italic',
					'Rubik Mono One' => 'Rubik Mono One:regular',
					'Ruda' => 'Ruda:regular,700,900',
					'Rufina' => 'Rufina:regular,700',
					'Ruge Boogie' => 'Ruge Boogie:regular',
					'Ruluko' => 'Ruluko:regular',
					'Rum Raisin' => 'Rum Raisin:regular',
					'Ruslan Display' => 'Ruslan Display:regular',
					'Russo One' => 'Russo One:regular',
					'Ruthie' => 'Ruthie:regular',
					'Rye' => 'Rye:regular',
					'Sacramento' => 'Sacramento:regular',
					'Sahitya' => 'Sahitya:regular,700',
					'Sail' => 'Sail:regular',
					'Salsa' => 'Salsa:regular',
					'Sanchez' => 'Sanchez:regular,italic',
					'Sancreek' => 'Sancreek:regular',
					'Sansita' => 'Sansita:regular,italic,700,700italic,800,800italic,900,900italic',
					'Sarala' => 'Sarala:regular,700',
					'Sarina' => 'Sarina:regular',
					'Sarpanch' => 'Sarpanch:regular,500,600,700,800,900',
					'Satisfy' => 'Satisfy:regular',
					'Scada' => 'Scada:regular,italic,700,700italic',
					'Scheherazade' => 'Scheherazade:regular,700',
					'Schoolbell' => 'Schoolbell:regular',
					'Scope One' => 'Scope One:regular',
					'Seaweed Script' => 'Seaweed Script:regular',
					'Secular One' => 'Secular One:regular',
					'Sevillana' => 'Sevillana:regular',
					'Seymour One' => 'Seymour One:regular',
					'Shadows Into Light' => 'Shadows Into Light:regular',
					'Shadows Into Light Two' => 'Shadows Into Light Two:regular',
					'Shanti' => 'Shanti:regular',
					'Share' => 'Share:regular,italic,700,700italic',
					'Share Tech' => 'Share Tech:regular',
					'Share Tech Mono' => 'Share Tech Mono:regular',
					'Shojumaru' => 'Shojumaru:regular',
					'Short Stack' => 'Short Stack:regular',
					'Shrikhand' => 'Shrikhand:regular',
					'Siemreap' => 'Siemreap:regular',
					'Sigmar One' => 'Sigmar One:regular',
					'Signika' => 'Signika:300,regular,600,700',
					'Signika Negative' => 'Signika Negative:300,regular,600,700',
					'Simonetta' => 'Simonetta:regular,italic,900,900italic',
					'Sintony' => 'Sintony:regular,700',
					'Sirin Stencil' => 'Sirin Stencil:regular',
					'Six Caps' => 'Six Caps:regular',
					'Skranji' => 'Skranji:regular,700',
					'Slabo 13px' => 'Slabo 13px:regular',
					'Slabo 27px' => 'Slabo 27px:regular',
					'Slackey' => 'Slackey:regular',
					'Smokum' => 'Smokum:regular',
					'Smythe' => 'Smythe:regular',
					'Sniglet' => 'Sniglet:regular,800',
					'Snippet' => 'Snippet:regular',
					'Snowburst One' => 'Snowburst One:regular',
					'Sofadi One' => 'Sofadi One:regular',
					'Sofia' => 'Sofia:regular',
					'Sonsie One' => 'Sonsie One:regular',
					'Sorts Mill Goudy' => 'Sorts Mill Goudy:regular,italic',
					'Source Code Pro' => 'Source Code Pro:200,300,regular,500,600,700,900',
					'Source Sans Pro' => 'Source Sans Pro:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,900,900italic',
					'Source Serif Pro' => 'Source Serif Pro:regular,600,700',
					'Space Mono' => 'Space Mono:regular,italic,700,700italic',
					'Special Elite' => 'Special Elite:regular',
					'Spectral' => 'Spectral:200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic',
					'Spicy Rice' => 'Spicy Rice:regular',
					'Spinnaker' => 'Spinnaker:regular',
					'Spirax' => 'Spirax:regular',
					'Squada One' => 'Squada One:regular',
					'Sree Krushnadevaraya' => 'Sree Krushnadevaraya:regular',
					'Sriracha' => 'Sriracha:regular',
					'Stalemate' => 'Stalemate:regular',
					'Stalinist One' => 'Stalinist One:regular',
					'Stardos Stencil' => 'Stardos Stencil:regular,700',
					'Stint Ultra Condensed' => 'Stint Ultra Condensed:regular',
					'Stint Ultra Expanded' => 'Stint Ultra Expanded:regular',
					'Stoke' => 'Stoke:300,regular',
					'Strait' => 'Strait:regular',
					'Sue Ellen Francisco' => 'Sue Ellen Francisco:regular',
					'Suez One' => 'Suez One:regular',
					'Sumana' => 'Sumana:regular,700',
					'Sunshiney' => 'Sunshiney:regular',
					'Supermercado One' => 'Supermercado One:regular',
					'Sura' => 'Sura:regular,700',
					'Suranna' => 'Suranna:regular',
					'Suravaram' => 'Suravaram:regular',
					'Suwannaphum' => 'Suwannaphum:regular',
					'Swanky and Moo Moo' => 'Swanky and Moo Moo:regular',
					'Syncopate' => 'Syncopate:regular,700',
					'Tangerine' => 'Tangerine:regular,700',
					'Taprom' => 'Taprom:regular',
					'Tauri' => 'Tauri:regular',
					'Taviraj' => 'Taviraj:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Teko' => 'Teko:300,regular,500,600,700',
					'Telex' => 'Telex:regular',
					'Tenali Ramakrishna' => 'Tenali Ramakrishna:regular',
					'Tenor Sans' => 'Tenor Sans:regular',
					'Text Me One' => 'Text Me One:regular',
					'The Girl Next Door' => 'The Girl Next Door:regular',
					'Tienne' => 'Tienne:regular,700,900',
					'Tillana' => 'Tillana:regular,500,600,700,800',
					'Timmana' => 'Timmana:regular',
					'Tinos' => 'Tinos:regular,italic,700,700italic',
					'Titan One' => 'Titan One:regular',
					'Titillium Web' => 'Titillium Web:200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,900',
					'Trade Winds' => 'Trade Winds:regular',
					'Trirong' => 'Trirong:100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic',
					'Trocchi' => 'Trocchi:regular',
					'Trochut' => 'Trochut:regular,italic,700',
					'Trykker' => 'Trykker:regular',
					'Tulpen One' => 'Tulpen One:regular',
					'Ubuntu' => 'Ubuntu:300,300italic,regular,italic,500,500italic,700,700italic',
					'Ubuntu Condensed' => 'Ubuntu Condensed:regular',
					'Ubuntu Mono' => 'Ubuntu Mono:regular,italic,700,700italic',
					'Ultra' => 'Ultra:regular',
					'Uncial Antiqua' => 'Uncial Antiqua:regular',
					'Underdog' => 'Underdog:regular',
					'Unica One' => 'Unica One:regular',
					'UnifrakturCook' => 'UnifrakturCook:700',
					'UnifrakturMaguntia' => 'UnifrakturMaguntia:regular',
					'Unkempt' => 'Unkempt:regular,700',

					'Unlock' => 'Unlock:regular',
					'Unna' => 'Unna:regular,italic,700,700italic',
					'VT323' => 'VT323:regular',
					'Vampiro One' => 'Vampiro One:regular',
					'Varela' => 'Varela:regular',
					'Varela Round' => 'Varela Round:regular',
					'Vast Shadow' => 'Vast Shadow:regular',
					'Vesper Libre' => 'Vesper Libre:regular,500,700,900',
					'Vibur' => 'Vibur:regular',
					'Vidaloka' => 'Vidaloka:regular',
					'Viga' => 'Viga:regular',
					'Voces' => 'Voces:regular',
					'Volkhov' => 'Volkhov:regular,italic,700,700italic',
					'Vollkorn' => 'Vollkorn:regular,italic,700,700italic',
					'Voltaire' => 'Voltaire:regular',
					'Waiting for the Sunrise' => 'Waiting for the Sunrise:regular',
					'Wallpoet' => 'Wallpoet:regular',
					'Walter Turncoat' => 'Walter Turncoat:regular',
					'Warnes' => 'Warnes:regular',
					'Wellfleet' => 'Wellfleet:regular',
					'Wendy One' => 'Wendy One:regular',
					'Wire One' => 'Wire One:regular',
					'Work Sans' => 'Work Sans:100,200,300,regular,500,600,700,800,900',
					'Yanone Kaffeesatz' => 'Yanone Kaffeesatz:200,300,regular,700',
					'Yantramanav' => 'Yantramanav:100,300,regular,500,700,900',
					'Yatra One' => 'Yatra One:regular',
					'Yellowtail' => 'Yellowtail:regular',
					'Yeseva One' => 'Yeseva One:regular',
					'Yesteryear' => 'Yesteryear:regular',
					'Yrsa' => 'Yrsa:300,regular,500,600,700',
					'Zeyada' => 'Zeyada:regular',
					'Zilla Slab' => 'Zilla Slab:300,300italic,regular,italic,500,500italic,600,600italic,700,700italic'
				)
			);
}
endif;

if ( ! function_exists( 'wppm_custom_meta' ) ) :
	function wppm_custom_meta() {
		$meta = array();
		$meta['author'] = sprintf( '<a href="%1$s"><span class="vcard author %2$s" itemprop="name"><span class="fn">%2$s</span></span></a>',
			esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
			get_the_author()
		);
		$meta['date'] = sprintf( '<span class="posted-on"><time itemprop="datePublished" class="entry-date updated" datetime="%s">%s</time></span>', get_the_date( DATE_W3C ), get_the_date() );
		$meta['date_modified'] = sprintf( '<span class="updated-on"><meta itemprop="dateModified" content="%s">%s %s</span>', get_the_modified_date( DATE_W3C ), get_the_modified_date(), get_the_modified_date('H:i:s') );

		// Comment link
		$num_comments = get_comments_number();
		$meta['comments'] = '';
		if ( comments_open() && ( $num_comments >= 1 ) ) {
			$meta['comments'] = sprintf( '<a href="%1$s" class="post-comment" title="%2$s">%3$s</a>',
				esc_url( get_comments_link() ),
				sprintf( __( 'Comment on %s', 'newsplus' ), esc_attr( get_the_title() ) ),
				$num_comments
			);
		}

		$meta['categories'] = get_the_category_list( _x( ', ', 'category items separator', 'wppm' ) );
		$meta['permalink'] = get_permalink();

		return $meta;
	}
endif;

if ( ! function_exists( 'wppm_generate_excerpt' ) ) :
	function wppm_generate_excerpt( $psource = 'excerpt', $allowed_tags = '', $content_filter = '', $cust_field_key = '', $excerpt_length = '10', $meta_box = '' ) {
		$excerpt_text = '';
		$post_id = get_the_id();

		if ( 'content' == $psource ) {
			$c = apply_filters( 'the_content', get_the_content() );
			$excerpt_text = wppm_short( $c, $excerpt_length, $allowed_tags );
		}

		elseif ( 'meta_box' == $psource ) {
			$meta_box_arr = get_post_meta( $post_id, $meta_box, true );
			if ( isset( $meta_box_arr ) && is_array( $meta_box_arr ) && isset( $meta_box_arr[ $cust_field_key ] ) && '' !== $meta_box_arr[ $cust_field_key ] ) {
				$excerpt_text = do_shortcode( $meta_box_arr[ $cust_field_key ] );
			}
			else {
				$excerpt_text = wppm_short( get_the_excerpt(), $excerpt_length, '' );
			}
		}

		elseif ( 'custom_field' == $psource ) {
			$excerpt_text = do_shortcode( get_post_meta( $post_id, $cust_field_key, true ) );
			if ( '' == $excerpt_text ) {
				$excerpt_text = wppm_short( get_the_excerpt(), $excerpt_length, '' );
			}
		}

		else {
			$excerpt_text = wppm_short( get_the_excerpt(), $excerpt_length, '' );
		}

		if ( $content_filter && ( 'meta_box' == $psource || 'custom_field' == $psource ) ) {
			$excerpt_text = apply_filters( 'the_content', $excerpt_text );
		}

		return $excerpt_text;
	}
endif;

if ( ! function_exists( 'wppm_generate_title' ) ) :
	function wppm_generate_title( $hsource = 'title', $h_cust_field_key = '', $h_length = '', $h_meta_box = '', $more = null ) {
		$post_id = get_the_id();
		$title_text = get_the_title( $post_id );

		if ( 'meta_box' == $hsource ) {
			$meta_box_arr = get_post_meta( $post_id, $h_meta_box, true );
			if ( isset( $meta_box_arr ) && is_array( $meta_box_arr ) && isset( $meta_box_arr[ $h_cust_field_key ] ) && '' !==  $meta_box_arr[ $h_cust_field_key ] ) {
				$title_text = $meta_box_arr[ $h_cust_field_key ];
			}
		} elseif ( 'custom_field' == $hsource ) {
			$cust_field_title = get_post_meta( $post_id, $h_cust_field_key, true );
			if ( isset( $cust_field_title ) && '' !== $cust_field_title ) {
				$title_text = $cust_field_title;
			}
		}

		if ( '' == $h_length ) {
			return $title_text;
		} else {
			return wp_trim_words( $title_text, $h_length, $more );
		}
	}
endif;