<?php
/*
 * @author    ThemePunch <info@themepunch.com>
 * @link      http://www.themepunch.com/
 * @copyright 2024 ThemePunch
*/

if(!defined('ABSPATH')) exit();

class SrWeatherFront extends RevSliderFunctions {

	private $slug;
	private $script_enqueued = false;
	private $api_key = false;

	public function __construct($slug){
		$key = get_option('revslider_weather_addon');
		if(is_string($key) && !empty($key)) $this->api_key = str_replace('revslider-weather-addon-api=', '', $key);

		global $SR_GLOBALS;
		$this->slug = str_replace(array('revslider-', '-addon'), '', $slug);
		if($this->get_val($SR_GLOBALS, 'front_version') === 7){
			add_filter('revslider_add_slider_base', array($this, 'enqueue_header_scripts'), 10, 1);
			add_filter('sr_get_full_slider_JSON', array($this, 'add_modal_scripts'), 10, 2);
			add_filter('revslider_export_html_file_inclusion', array($this, 'add_addon_files'), 10, 2);
		}
		if($this->get_val($_GET, 'page') === 'revslider'){
			add_action('admin_footer', array($this, 'add_header_scripts_return'));
		}
		add_action('rest_api_init', array($this, 'init_rest_api'));
	}

	public function add_modal_scripts($obj, $slider){
		if(!$this->is_in_use($slider)) return $obj;
		$list = $this->get_script_list();
		if(empty($list)) return $obj;
		foreach($list ?? [] as $handle => $script){
			$obj['addOns'][$handle] = $script;
		}

		return $obj;
	}

	public function add_header_scripts($script){
		echo $this->add_header_scripts_return(false);
	}

	public function add_header_scripts_return($tags = ''){
		if($tags !== false){
			if($this->script_enqueued) return;
			$this->script_enqueued = true;
		}

		$list = $this->get_script_list();
		if(empty($list)) return '';

		$tab	= ($tags !== false) ? '' : '	';
		$nl		= (count($list) > 1 || $tags === false) ? "\n" : '';
		$html	= '';
		$html	.= ($tags !== false) ? "<script>".$nl : '';
		foreach($list ?? [] as $handle => $script){
			$html .= $tab.'SR7.E.resources.'.$handle.' = "'. $script .'";'.$nl;
		}
		$html	.= ($tags !== false) ? "</script>" . "\n" : '';

		if($tags === false) return $html;

		echo $html;
	}

	public function get_script_list(){
		$min = file_exists(REV_ADDON_WEATHER_PATH . 'public/js/' . $this->slug . '.js') ? '' : '.min';
		return array(
			'weather' => REV_ADDON_WEATHER_URL.'public/js/' . $this->slug . $min . '.js',
			'cssweather' => REV_ADDON_WEATHER_URL.'public/css/' . $this->slug . '.css'
		);
	}

	public function enqueue_header_scripts($slider){
		if($this->script_enqueued) return $slider;
		if(empty($slider)) return $slider;

		if($this->is_in_use($slider)) $this->enqueue_scripts();

		return $slider;
	}

	public function is_in_use($slider){
		if(empty($slider)) return false;

		// check if we are an v7 slider
		if($this->get_val($slider, array('settings', 'migrated'), false) !== false && $this->get_val($slider, array('settings', 'addOns', $this->slug, 'u')) === true) return true;
		if($this->get_val($slider, array('params', 'migrated'), false) !== false && $this->get_val($slider, array('params', 'addOns', $this->slug, 'u')) === true) return true;

		// check if we are v6
		if($this->get_val($slider, array('slider_params', 'addOns', 'revslider-'.$this->slug.'-addon', 'enable'), false) === true) return true;
		if($this->get_val($slider, array('params', 'addOns', 'revslider-'.$this->slug.'-addon', 'enable'), false) === true) return true;

		// check v7 if false is set
		if($this->get_val($slider, array('settings', 'migrated'), false) !== false && $this->get_val($slider, array('settings', 'addOns', $this->slug, 'u')) === false) return false;
		if($this->get_val($slider, array('params', 'migrated'), false) !== false && $this->get_val($slider, array('params', 'addOns', $this->slug, 'u')) === false) return false;

		// check v6 if false is set
		if($this->get_val($slider, array('slider_params', 'addOns', 'revslider-'.$this->slug.'-addon', 'enable'), 'unset') === false) return false;
		if($this->get_val($slider, array('params', 'addOns', 'revslider-'.$this->slug.'-addon', 'enable'), 'unset') === false) return false;

		//check if we are v6, and maybe some deeper element needs the addon
		$json = json_encode($slider, true);
		$return = (strpos($json, 'revslider-'.$this->slug.'-addon') !== false) ? true : false;
		unset($json);

		return $return;
	}

	public function enqueue_scripts(){
		add_action('revslider_pre_add_js', array($this, 'add_header_scripts'));
		$this->script_enqueued = true;
		$min = file_exists(REV_ADDON_WEATHER_PATH . 'public/js/' . $this->slug . '.js') ? '' : '.min';
		wp_enqueue_script('revslider-'.$this->slug.'-addon', REV_ADDON_WEATHER_URL . "public/js/" . $this->slug . $min . ".js", '', RS_REVISION, array('strategy' => 'async'));
		wp_enqueue_style('revslider-'.$this->slug.'-addon', REV_ADDON_WEATHER_URL . "public/css/" . $this->slug . ".css", '', RS_REVISION);
	}

	public function add_addon_files($html, $export){
		$output = $export->slider_output;
		$addOn = $this->is_in_use($output->slider);
		if(empty($addOn)) return $html;

		$list = $this->get_script_list();
		if(empty($list)) return '';
		
		foreach($list ?? [] as $handle => $script){
			$info = pathinfo($script);
			$extension = (isset($info['extension'])) ? $info['extension'] : 'js';
			$export_path = 'revslider-'.$this->slug.'-addon/public/'.$extension.'/';

			$script_path = str_replace(REV_ADDON_WEATHER_URL, REV_ADDON_WEATHER_PATH, $script);
			if(!$export->usepcl){
				$export->zip->addFile($script_path,  $export_path . $handle . '.'.$extension);
			}else{
				$export->pclzip->add($script_path, PCLZIP_OPT_REMOVE_PATH, REV_ADDON_WEATHER_PATH.'public/'.$extension.'/', PCLZIP_OPT_ADD_PATH, $export_path);
			}
			$html = str_replace(array($script, str_replace('/', '\/', $script)), array($export_path . $handle . '.'.$extension, str_replace('/', '\/', $export_path . $handle . '.'.$extension)), $html);
		}

		return $html;
	}

	/**
	 * Init the REST API
	 **/
	public function init_rest_api()
	{
		register_rest_route('sliderrevolution', '/weather/l/(?P<l>[^/]+)/unit/(?P<unit>[^/]+)/id/(?P<id>\d+)', array(
			'methods'				=> WP_REST_SERVER::READABLE,
			'callback'				=> array($this, 'get_weather_object'),
			'permission_callback' => '__return_true',
		));
	}

	public function get_weather_object(WP_REST_Request $request )
	{
		$params = $request->get_url_params();
		$location = $this->get_val($params, 'l', false);
		if (!$location) {
			$response = array(
				'success'	=> false,
				'message'	=> _e('Location can not be empty', 'revslider-weather-addon')
			);
			return new WP_REST_Response($response, 409);
		}

		$unit = $this->get_val($params, 'unit', 'c');
		$unit = $unit === 'f' ? 'imperial' : 'metric';

		$refresh = $this->get_val($params, 'refresh', 0);

		//Get weather information dependent from Slider options
		$weather = $this->get_weather($location, $unit, $refresh);

		if (isset($weather->status_code)) {
			$response = array('success'	=> false, 'message'	=> json_encode($weather));
			return new WP_REST_Response($response, $weather->status_code);
		}

		$response = array('success'	=> true, 'message'	=> json_encode($weather));
		return new WP_REST_Response($response, 200);
	}

	/**
	 * Connects to Weatherbit.io and collects Weather Info
	 * @param string $location
	 * @param string $unit
	 * @param int $refresh
	 * @return mixed
	 */
	public function get_weather($location, $unit, $refresh) {
		$revslider_weather = RevAddOnWeatherOWM::get_weather_infos($location, $unit, $this->api_key, $refresh);

		return ($this->get_val($revslider_weather, 'error', false))  ? false : $revslider_weather;
	}

}
