<?php
/**
 * Jalali (Shamsi) dates for frontend display.
 *
 * - Database stays Gregorian (no schema/sorting break)
 * - Only affects human-facing outputs (not ISO / machine formats)
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Convert Gregorian to Jalali.
 *
 * @return array{0:int,1:int,2:int} [jy, jm, jd]
 */
function disto_child_gregorian_to_jalali( int $gy, int $gm, int $gd ): array {
	$g_d_m = array( 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 );
	$gy2   = ( $gm > 2 ) ? ( $gy + 1 ) : $gy;
	$days  = 355666 + ( 365 * $gy ) + (int) floor( ( $gy2 + 3 ) / 4 ) - (int) floor( ( $gy2 + 99 ) / 100 ) + (int) floor( ( $gy2 + 399 ) / 400 ) + $gd + $g_d_m[ $gm - 1 ];
	$jy    = -1595 + ( 33 * (int) floor( $days / 12053 ) );
	$days  = $days % 12053;
	$jy   += 4 * (int) floor( $days / 1461 );
	$days  = $days % 1461;
	if ( $days > 365 ) {
		$jy  += (int) floor( ( $days - 1 ) / 365 );
		$days = ( $days - 1 ) % 365;
	}
	if ( $days < 186 ) {
		$jm = 1 + (int) floor( $days / 31 );
		$jd = 1 + ( $days % 31 );
	} else {
		$jm = 7 + (int) floor( ( $days - 186 ) / 30 );
		$jd = 1 + ( ( $days - 186 ) % 30 );
	}
	return array( (int) $jy, (int) $jm, (int) $jd );
}

function disto_child_jalali_month_name( int $jm ): string {
	$names = array(
		1  => 'فروردین',
		2  => 'اردیبهشت',
		3  => 'خرداد',
		4  => 'تیر',
		5  => 'مرداد',
		6  => 'شهریور',
		7  => 'مهر',
		8  => 'آبان',
		9  => 'آذر',
		10 => 'دی',
		11 => 'بهمن',
		12 => 'اسفند',
	);
	return $names[ $jm ] ?? (string) $jm;
}

function disto_child_jalali_month_short( int $jm ): string {
	$names = array(
		1  => 'فرو',
		2  => 'ارد',
		3  => 'خرد',
		4  => 'تیر',
		5  => 'مرد',
		6  => 'شهر',
		7  => 'مهر',
		8  => 'آبا',
		9  => 'آذر',
		10 => 'دی',
		11 => 'بهم',
		12 => 'اسف',
	);
	return $names[ $jm ] ?? (string) $jm;
}

function disto_child_weekday_name( int $w ): string {
	$names = array(
		0 => 'یکشنبه',
		1 => 'دوشنبه',
		2 => 'سه‌شنبه',
		3 => 'چهارشنبه',
		4 => 'پنجشنبه',
		5 => 'جمعه',
		6 => 'شنبه',
	);
	return $names[ $w ] ?? '';
}

function disto_child_weekday_short( int $w ): string {
	$names = array(
		0 => 'یک',
		1 => 'دو',
		2 => 'سه',
		3 => 'چه',
		4 => 'پن',
		5 => 'جم',
		6 => 'شن',
	);
	return $names[ $w ] ?? '';
}

/**
 * Jalali date formatting (subset of PHP date tokens commonly used in themes).
 */
function disto_child_jdate( string $format, int $timestamp ): string {
	$gy = (int) gmdate( 'Y', $timestamp + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
	$gm = (int) gmdate( 'n', $timestamp + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
	$gd = (int) gmdate( 'j', $timestamp + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );

	$time_local = $timestamp + (int) ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS );
	$w          = (int) gmdate( 'w', $time_local );

	list( $jy, $jm, $jd ) = disto_child_gregorian_to_jalali( $gy, $gm, $gd );

	$replacements = array(
		'Y' => (string) $jy,
		'y' => substr( (string) $jy, -2 ),
		'm' => str_pad( (string) $jm, 2, '0', STR_PAD_LEFT ),
		'n' => (string) $jm,
		'd' => str_pad( (string) $jd, 2, '0', STR_PAD_LEFT ),
		'j' => (string) $jd,
		'F' => disto_child_jalali_month_name( $jm ),
		'M' => disto_child_jalali_month_short( $jm ),
		'l' => disto_child_weekday_name( $w ),
		'D' => disto_child_weekday_short( $w ),
	);

	// Replace only supported tokens, keep others (including time tokens) from WP formatting.
	$out = '';
	$len = strlen( $format );
	for ( $i = 0; $i < $len; $i++ ) {
		$ch = $format[ $i ];
		if ( $ch === '\\' && ( $i + 1 ) < $len ) {
			$out .= $format[ $i + 1 ];
			$i++;
			continue;
		}
		if ( isset( $replacements[ $ch ] ) ) {
			$out .= $replacements[ $ch ];
		} else {
			$out .= gmdate( $ch, $time_local ); // falls back for time / punctuation.
		}
	}

	return $out;
}

function disto_child_should_jalalify_format( ?string $format ): bool {
	if ( $format === null || $format === '' ) {
		return true;
	}
	$format = trim( $format );
	if ( $format === 'c' || $format === 'r' || $format === 'U' ) {
		return false; // machine formats
	}
	// ISO-like formats should remain Gregorian for schema/feeds.
	if ( preg_match( '/Y-?m-?d(T|\\s)/', $format ) ) {
		return false;
	}
	return true;
}

function disto_child_is_safe_context_for_jalali(): bool {
	if ( is_admin() ) {
		return false;
	}
	if ( function_exists( 'wp_is_json_request' ) && wp_is_json_request() ) {
		return false;
	}
	if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
		return false;
	}
	if ( is_feed() ) {
		return false;
	}
	return true;
}

/**
 * Post dates.
 */
add_filter( 'get_the_date', function ( $the_date, $format, $post ) {
	if ( ! disto_child_is_safe_context_for_jalali() ) {
		return $the_date;
	}
	if ( ! disto_child_should_jalalify_format( $format ) ) {
		return $the_date;
	}

	$timestamp = get_post_timestamp( $post );
	if ( ! $timestamp ) {
		return $the_date;
	}

	$fmt = $format ?: get_option( 'date_format' );
	return disto_child_jdate( $fmt, (int) $timestamp );
}, 20, 3 );

add_filter( 'get_the_time', function ( $the_time, $format, $post ) {
	if ( ! disto_child_is_safe_context_for_jalali() ) {
		return $the_time;
	}
	// Time-only formats are fine; but if a full ISO is requested, do not touch.
	if ( ! disto_child_should_jalalify_format( $format ) ) {
		return $the_time;
	}

	$timestamp = get_post_timestamp( $post );
	if ( ! $timestamp ) {
		return $the_time;
	}

	$fmt = $format ?: get_option( 'time_format' );
	return disto_child_jdate( $fmt, (int) $timestamp );
}, 20, 3 );

/**
 * Comment dates.
 */
add_filter( 'get_comment_date', function ( $date, $format, $comment ) {
	if ( ! disto_child_is_safe_context_for_jalali() ) {
		return $date;
	}
	if ( ! disto_child_should_jalalify_format( $format ) ) {
		return $date;
	}
	$timestamp = get_comment_date( 'U', $comment );
	if ( ! $timestamp ) {
		return $date;
	}
	$fmt = $format ?: get_option( 'date_format' );
	return disto_child_jdate( $fmt, (int) $timestamp );
}, 20, 3 );

add_filter( 'get_comment_time', function ( $time, $format, $comment ) {
	if ( ! disto_child_is_safe_context_for_jalali() ) {
		return $time;
	}
	if ( ! disto_child_should_jalalify_format( $format ) ) {
		return $time;
	}
	$timestamp = get_comment_date( 'U', $comment );
	if ( ! $timestamp ) {
		return $time;
	}
	$fmt = $format ?: get_option( 'time_format' );
	return disto_child_jdate( $fmt, (int) $timestamp );
}, 20, 3 );

/**
 * WooCommerce formatted datetime (human-facing).
 */
add_filter( 'woocommerce_format_datetime', function ( $formatted, $datetime, $format ) {
	if ( ! disto_child_is_safe_context_for_jalali() ) {
		return $formatted;
	}
	if ( ! $datetime instanceof DateTimeInterface ) {
		return $formatted;
	}
	if ( ! disto_child_should_jalalify_format( $format ) ) {
		return $formatted;
	}
	$fmt = $format ?: get_option( 'date_format' );
	return disto_child_jdate( $fmt, (int) $datetime->getTimestamp() );
}, 20, 3 );

