<?php
/**
 * Kalameh font CSS endpoint.
 *
 * This serves a real stylesheet via admin-ajax.php so we can dynamically
 * locate the font files inside a plugin directory without inline CSS.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Get the CSS URL for Kalameh @font-face.
 */
function disto_child_kalameh_css_url(): string {
	return admin_url( 'admin-ajax.php?action=disto_child_kalameh_css' );
}

/**
 * Version string for Kalameh CSS (cache-busting).
 */
function disto_child_kalameh_css_version(): string {
	// Changes when child theme updates; can also be filtered if you want.
	$ver = DISTO_CHILD_VERSION;
	return (string) apply_filters( 'disto_child_kalameh_css_version', $ver );
}

/**
 * Locate Kalameh font directory.
 *
 * Client requirement: fonts live in a `font` folder inside the Disto theme.
 *   Example: get_template_directory() . '/font/KalamehWeb-Regular.woff2'
 *
 * You can still override this via:
 *   add_filter( 'disto_child_kalameh_base_dir', fn() => ABSPATH . 'custom-font-path' );
 *   add_filter( 'disto_child_kalameh_base_url', fn() => home_url( '/custom-font-path' ) );
 */
function disto_child_kalameh_base_dir(): ?string {
	$filtered = apply_filters( 'disto_child_kalameh_base_dir', null );
	if ( is_string( $filtered ) && $filtered !== '' ) {
		return rtrim( wp_normalize_path( $filtered ), '/' );
	}

	// 1) Try /font (some installs use this).
	$root_font_dir = wp_normalize_path( get_template_directory() . '/font' );
	if ( is_dir( $root_font_dir ) ) {
		return rtrim( $root_font_dir, '/' );
	}

	// 2) Try /fonts (very common naming, and matches client's wording).
	$root_fonts_dir = wp_normalize_path( get_template_directory() . '/fonts' );
	if ( is_dir( $root_fonts_dir ) ) {
		return rtrim( $root_fonts_dir, '/' );
	}

	return null;
}

function disto_child_kalameh_base_url(): ?string {
	$filtered = apply_filters( 'disto_child_kalameh_base_url', null );
	if ( is_string( $filtered ) && $filtered !== '' ) {
		return rtrim( $filtered, '/' );
	}

	$base_dir = disto_child_kalameh_base_dir();
	if ( ! $base_dir ) {
		return null;
	}

	// Match whichever directory actually exists in the parent theme.
	// If /font exists, its URL will be /font; if /fonts exists, its URL will be /fonts.
	$theme_dir  = wp_normalize_path( get_template_directory() );
	$theme_uri  = untrailingslashit( get_template_directory_uri() );

	// Remove theme dir prefix from base_dir to get relative path.
	if ( strpos( $base_dir, $theme_dir ) === 0 ) {
		$relative = ltrim( substr( $base_dir, strlen( $theme_dir ) ), '/\\' );
		return $theme_uri . '/' . $relative;
	}

	return null;
}

/**
 * Pick best font file URL for a given base and stem.
 */
function disto_child_font_src( string $base_url, string $base_dir, string $stem ): array {
	$base_url = rtrim( $base_url, '/' );
	$base_dir = rtrim( $base_dir, '/\\' );

	$woff2_path = $base_dir . '/' . $stem . '.woff2';
	$woff_path  = $base_dir . '/' . $stem . '.woff';

	$src = array();
	if ( file_exists( $woff2_path ) ) {
		$src[] = 'url("' . esc_url_raw( $base_url . '/' . rawurlencode( $stem ) . '.woff2' ) . '") format("woff2")';
	}
	if ( file_exists( $woff_path ) ) {
		$src[] = 'url("' . esc_url_raw( $base_url . '/' . rawurlencode( $stem ) . '.woff' ) . '") format("woff")';
	}
	return $src;
}

/**
 * Render the Kalameh @font-face CSS.
 */
function disto_child_render_kalameh_css(): string {
	$base_url = disto_child_kalameh_base_url();
	$base_dir = disto_child_kalameh_base_dir();
	if ( ! $base_url || ! $base_dir ) {
		// Safe fallback: still declare family so typography CSS doesn't break layout.
		return "/* Kalameh fonts not found. Set disto_child_kalameh_base_url filter. */\n";
	}

	// Common filename stems for Kalameh / KalamehWeb.
	$weights = array(
		100 => array(
			'KalamehWeb-Thin',
			'Kalameh-Thin',
			'Kalameh-Thin-FD',
			'Kalameh-ThinFd',
			'Kalameh-Thin-WF',
			'Kalameh-ThinWF',
		),
		300 => array(
			'KalamehWeb-Light',
			'Kalameh-Light',
			'Kalameh-Light-FD',
			'Kalameh-LightFd',
			'Kalameh-LightFd-WF',
			'Kalameh-Light-WF',
			'Kalameh-LightWF',
		),
		400 => array(
			'KalamehWeb',
			'KalamehWeb-Regular',
			'Kalameh-Regular',
			'Kalameh',
			'Kalameh-Regular-FD',
			'Kalameh-RegularFd',
			'Kalameh-Regular-WF',
			'Kalameh-RegularWF',
		),
		500 => array(
			'KalamehWeb-Medium',
			'Kalameh-Medium',
			'Kalameh-Medium-FD',
			'Kalameh-MediumFd',
			'Kalameh-Medium-WF',
			'Kalameh-MediumWF',
		),
		600 => array(
			'KalamehWeb-SemiBold',
			'Kalameh-SemiBold',
			'Kalameh-Semibold',
			'Kalameh-SemiBold-FD',
			'Kalameh-SemiBoldFd',
			'Kalameh-SemiBold-WF',
			'Kalameh-SemiBoldWF',
		),
		700 => array(
			'KalamehWeb-Bold',
			'Kalameh-Bold',
			'Kalameh-Bold-FD',
			'Kalameh-BoldFd',
			'Kalameh-Bold-WF',
			'Kalameh-BoldWF',
		),
	);

	$out = array();
	$out[] = '/* Kalameh font-face (served by disto-child) */';

	foreach ( $weights as $weight => $stems ) {
		$src = array();
		foreach ( $stems as $stem ) {
			$src = disto_child_font_src( $base_url, $base_dir, $stem );
			if ( ! empty( $src ) ) {
				break;
			}
		}

		if ( empty( $src ) ) {
			continue;
		}

		$out[] = '@font-face {';
		$out[] = "  font-family: 'KalamehWeb';";
		$out[] = '  src: ' . implode( ",\n       ", $src ) . ';';
		$out[] = '  font-weight: ' . (int) $weight . ';';
		$out[] = '  font-style: normal;';
		$out[] = '  font-display: swap;';
		$out[] = '}';
	}

	$out[] = '';

	return implode( "\n", $out );
}

/**
 * AJAX handler: outputs the CSS.
 */
function disto_child_kalameh_css_ajax() {
	// Cache for performance.
	$cache_key = 'disto_child_kalameh_css_' . md5( (string) disto_child_kalameh_base_url() . '|' . DISTO_CHILD_VERSION );
	$css       = get_transient( $cache_key );

	if ( ! is_string( $css ) ) {
		$css = disto_child_render_kalameh_css();
		set_transient( $cache_key, $css, DAY_IN_SECONDS );
	}

	nocache_headers();
	header( 'Content-Type: text/css; charset=UTF-8' );
	echo $css; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	wp_die();
}
add_action( 'wp_ajax_disto_child_kalameh_css', 'disto_child_kalameh_css_ajax' );
add_action( 'wp_ajax_nopriv_disto_child_kalameh_css', 'disto_child_kalameh_css_ajax' );

