<?php
/**
 * WooCommerce Products Widget (Elementor Widget)
 *
 * ویجت نمایش محصولات ووکامرس با طراحی مدرن و ریسپانسیو
 * - RTL friendly (Persian sites)
 * - WooCommerce integration
 * - Fully customizable
 * - Responsive design
 * - Hover effects with customizable border radius
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

if ( ! class_exists( 'WooCommerce' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;

class Disto_WooCommerce_Products_Widget extends Widget_Base {

	public function get_name() {
		return 'disto_woocommerce_products';
	}

	public function get_title() {
		return esc_html__( 'محصولات ووکامرس', 'disto' );
	}

	public function get_icon() {
		return 'eicon-products';
	}

	public function get_categories() {
		return array( 'disto-elements' );
	}

	public function get_style_depends() {
		return array( 'disto-child-woocommerce-products' );
	}

	public function get_script_depends() {
		if ( class_exists( 'WooCommerce' ) ) {
			return array( 'disto-child-woocommerce-products' );
		}
		return array();
	}

	/**
	 * فرمت قیمت با واحد پول صحیح
	 */
	private function format_price_html( $price_html ) {
		if ( empty( $price_html ) ) {
			return $price_html;
		}

		$currency = get_woocommerce_currency();
		$currency_symbol = get_woocommerce_currency_symbol();
		
		// اگر واحد پول ریال یا تومان است، نماد را جایگزین می‌کنیم
		if ( $currency === 'IRR' ) {
			// جایگزینی نماد با "ریال"
			$price_html = str_replace( $currency_symbol, 'ریال', $price_html );
			// همچنین اگر نماد در HTML به صورت entity یا دیگر فرمت‌ها باشد
			$price_html = preg_replace( '/\s*' . preg_quote( html_entity_decode( $currency_symbol ), '/' ) . '\s*/u', ' ریال ', $price_html );
		} elseif ( $currency === 'IRT' || $currency === 'IRHT' ) {
			// جایگزینی نماد با "تومان"
			$price_html = str_replace( $currency_symbol, 'تومان', $price_html );
			// همچنین اگر نماد در HTML به صورت entity یا دیگر فرمت‌ها باشد
			$price_html = preg_replace( '/\s*' . preg_quote( html_entity_decode( $currency_symbol ), '/' ) . '\s*/u', ' تومان ', $price_html );
		}
		
		// پاکسازی فاصله‌های اضافی
		$price_html = preg_replace( '/\s+/u', ' ', $price_html );
		
		return $price_html;
	}

	protected function register_controls() {
		// بخش تنظیمات محتوا
		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'تنظیمات محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		// انتخاب دسته‌بندی محصولات
		$product_categories = array( 0 => esc_html__( 'همه محصولات', 'disto' ) );
		if ( taxonomy_exists( 'product_cat' ) ) {
			$terms = get_terms(
				array(
					'taxonomy'   => 'product_cat',
					'hide_empty' => false,
				)
			);
			if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
				foreach ( $terms as $term ) {
					$product_categories[ (int) $term->term_id ] = $term->name;
				}
			}
		}

		$this->add_control(
			'product_category',
			array(
				'label'   => esc_html__( 'دسته‌بندی محصولات', 'disto' ),
				'type'    => Controls_Manager::SELECT2,
				'options' => $product_categories,
				'default' => 0,
			)
		);

		$this->add_control(
			'products_count',
			array(
				'label'   => esc_html__( 'تعداد محصولات', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 50,
				'step'    => 1,
				'default' => 8,
			)
		);

		$this->add_responsive_control(
			'products_per_row',
			array(
				'label'       => esc_html__( 'تعداد محصولات در هر ردیف', 'disto' ),
				'type'        => Controls_Manager::NUMBER,
				'min'         => 1,
				'max'         => 10,
				'step'        => 1,
				'default'     => 4,
				'tablet_default' => 2,
				'mobile_default' => 1,
				'description' => esc_html__( 'تعداد محصولاتی که در هر ردیف نمایش داده می‌شوند', 'disto' ),
			)
		);

		$this->add_control(
			'order_by',
			array(
				'label'   => esc_html__( 'مرتب‌سازی بر اساس', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => array(
					'date'       => esc_html__( 'تاریخ (جدیدترین)', 'disto' ),
					'title'      => esc_html__( 'عنوان', 'disto' ),
					'price'      => esc_html__( 'قیمت (کم به زیاد)', 'disto' ),
					'price-desc' => esc_html__( 'قیمت (زیاد به کم)', 'disto' ),
					'popularity' => esc_html__( 'محبوبیت', 'disto' ),
					'rating'     => esc_html__( 'امتیاز', 'disto' ),
					'rand'       => esc_html__( 'تصادفی', 'disto' ),
				),
			)
		);

		$this->add_control(
			'show_rating',
			array(
				'label'   => esc_html__( 'نمایش امتیاز', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_discount_badge',
			array(
				'label'   => esc_html__( 'نمایش برچسب تخفیف', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_add_to_cart',
			array(
				'label'   => esc_html__( 'نمایش دکمه افزودن به سبد خرید', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_favorite_btn',
			array(
				'label'   => esc_html__( 'نمایش دکمه علاقه‌مندی', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_stock_badge',
			array(
				'label'   => esc_html__( 'نمایش برچسب موجودی', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'cart_button_icon',
			array(
				'label'       => esc_html__( 'آیکون دکمه سبد خرید', 'disto' ),
				'type'        => Controls_Manager::ICONS,
				'default'     => array(
					'value'   => 'fas fa-shopping-cart',
					'library' => 'fa-solid',
				),
				'condition'   => array(
					'show_add_to_cart' => 'yes',
				),
			)
		);

		$this->add_control(
			'favorite_button_icon',
			array(
				'label'       => esc_html__( 'آیکون دکمه علاقه‌مندی', 'disto' ),
				'type'        => Controls_Manager::ICONS,
				'default'     => array(
					'value'   => 'fas fa-heart',
					'library' => 'fa-solid',
				),
				'condition'   => array(
					'show_favorite_btn' => 'yes',
				),
			)
		);

		$this->end_controls_section();

		// بخش تنظیمات پس‌زمینه
		$this->start_controls_section(
			'background_section',
			array(
				'label' => esc_html__( 'تنظیمات پس‌زمینه', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'widget_background',
				'label'    => esc_html__( 'پس‌زمینه ویجت', 'disto' ),
				'types'    => array( 'classic', 'gradient', 'image' ),
				'selector' => '{{WRAPPER}} .dwp-products-wrapper',
			)
		);

		$this->add_control(
			'background_overlay',
			array(
				'label'     => esc_html__( 'پوشش پس‌زمینه', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .dwp-products-wrapper::before' => 'background-color: {{VALUE}};',
				),
				'condition' => array(
					'widget_background_background' => array( 'classic', 'gradient', 'image' ),
				),
			)
		);

		$this->add_control(
			'background_overlay_opacity',
			array(
				'label'      => esc_html__( 'شفافیت پوشش', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 0,
						'max'  => 1,
						'step' => 0.1,
					),
				),
				'default'    => array(
					'size' => 0.5,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-products-wrapper::before' => 'opacity: {{SIZE}};',
				),
				'condition'  => array(
					'widget_background_background' => array( 'classic', 'gradient', 'image' ),
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل کارت محصول
		$this->start_controls_section(
			'card_style_section',
			array(
				'label' => esc_html__( 'استایل کارت محصول', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'card_background',
			array(
				'label'     => esc_html__( 'پس‌زمینه کارت', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-card' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'card_border_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 16,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-card' => 'border-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .dwp-product-image' => 'border-radius: {{SIZE}}{{UNIT}} {{SIZE}}{{UNIT}} 0 0;',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'card_shadow',
				'label'    => esc_html__( 'سایه کارت', 'disto' ),
				'selector' => '{{WRAPPER}} .dwp-product-card',
			)
		);

		$this->add_responsive_control(
			'card_padding',
			array(
				'label'      => esc_html__( 'فاصله داخلی کارت', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'cards_gap',
			array(
				'label'      => esc_html__( 'فاصله بین کارت‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 60,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 24,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-products-grid' => 'gap: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل Hover
		$this->start_controls_section(
			'hover_style_section',
			array(
				'label' => esc_html__( 'استایل Hover', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'enable_hover_effect',
			array(
				'label'   => esc_html__( 'فعال کردن افکت Hover', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'label_on' => esc_html__( 'فعال', 'disto' ),
				'label_off' => esc_html__( 'غیرفعال', 'disto' ),
			)
		);

		$this->add_control(
			'card_hover_border_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه‌ها در Hover', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 20,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-card:hover' => 'border-radius: {{SIZE}}{{UNIT}};',
				),
				'condition' => array(
					'enable_hover_effect' => 'yes',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'card_hover_shadow',
				'label'    => esc_html__( 'سایه کارت (Hover)', 'disto' ),
				'selector' => '{{WRAPPER}} .dwp-product-card:hover',
				'condition' => array(
					'enable_hover_effect' => 'yes',
				),
			)
		);

		$this->add_control(
			'hover_scale',
			array(
				'label'      => esc_html__( 'مقیاس Hover', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 1,
						'max'  => 1.2,
						'step' => 0.01,
					),
				),
				'default'    => array(
					'size' => 1.03,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-card:hover' => 'transform: scale({{SIZE}});',
				),
				'condition' => array(
					'enable_hover_effect' => 'yes',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل تصویر محصول
		$this->start_controls_section(
			'image_style_section',
			array(
				'label' => esc_html__( 'استایل تصویر محصول', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'image_height',
			array(
				'label'      => esc_html__( 'ارتفاع تصویر', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', '%', 'vh' ),
				'range'      => array(
					'px' => array(
						'min' => 100,
						'max' => 600,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 250,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-image-wrapper' => 'min-height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'image_border_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه‌های تصویر', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 16,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-image' => 'border-radius: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'image_object_fit',
			array(
				'label'   => esc_html__( 'نحوه نمایش تصویر', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'contain',
				'options' => array(
					'contain' => esc_html__( 'نمایش کامل (بدون کراپ)', 'disto' ),
					'cover'   => esc_html__( 'پر کردن فضا (با کراپ)', 'disto' ),
					'fill'    => esc_html__( 'پر کردن کامل', 'disto' ),
				),
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-image' => 'object-fit: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'image_object_position',
			array(
				'label'   => esc_html__( 'موقعیت تصویر', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'center',
				'options' => array(
					'center'      => esc_html__( 'مرکز', 'disto' ),
					'top'         => esc_html__( 'بالا', 'disto' ),
					'bottom'      => esc_html__( 'پایین', 'disto' ),
					'left'        => esc_html__( 'چپ', 'disto' ),
					'right'       => esc_html__( 'راست', 'disto' ),
					'top left'    => esc_html__( 'بالا چپ', 'disto' ),
					'top right'   => esc_html__( 'بالا راست', 'disto' ),
					'bottom left' => esc_html__( 'پایین چپ', 'disto' ),
					'bottom right' => esc_html__( 'پایین راست', 'disto' ),
				),
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-image' => 'object-position: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'image_background_color',
			array(
				'label'     => esc_html__( 'رنگ پس‌زمینه تصویر', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#000000',
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-image-wrapper' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .dwp-product-image' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'image_padding',
			array(
				'label'      => esc_html__( 'فاصله داخلی تصویر', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-image-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل عنوان محصول
		$this->start_controls_section(
			'title_style_section',
			array(
				'label' => esc_html__( 'استایل عنوان محصول', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'label'    => esc_html__( 'تایپوگرافی عنوان', 'disto' ),
				'selector' => '{{WRAPPER}} .dwp-product-title',
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'رنگ عنوان', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#1f2937',
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'title_margin',
			array(
				'label'      => esc_html__( 'فاصله عنوان', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل قیمت
		$this->start_controls_section(
			'price_style_section',
			array(
				'label' => esc_html__( 'استایل قیمت', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'price_typography',
				'label'    => esc_html__( 'تایپوگرافی قیمت', 'disto' ),
				'selector' => '{{WRAPPER}} .dwp-product-price',
			)
		);

		$this->add_control(
			'price_color',
			array(
				'label'     => esc_html__( 'رنگ قیمت', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#3b82f6',
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-price' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'sale_price_color',
			array(
				'label'     => esc_html__( 'رنگ قیمت تخفیف‌خورده', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ff6b6b',
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-price .price del' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل دکمه افزودن به سبد خرید
		$this->start_controls_section(
			'button_style_section',
			array(
				'label' => esc_html__( 'استایل دکمه افزودن به سبد', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'button_typography',
				'label'    => esc_html__( 'تایپوگرافی دکمه', 'disto' ),
				'selector' => '{{WRAPPER}} .dwp-add-to-cart-btn',
			)
		);

		$this->add_control(
			'button_background',
			array(
				'label'     => esc_html__( 'پس‌زمینه دکمه', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#3b82f6',
				'selectors' => array(
					'{{WRAPPER}} .dwp-add-to-cart-btn' => 'background: linear-gradient(135deg, {{VALUE}} 0%, {{VALUE}}dd 100%);',
				),
			)
		);

		$this->add_control(
			'button_color',
			array(
				'label'     => esc_html__( 'رنگ متن دکمه', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .dwp-add-to-cart-btn' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'button_hover_background',
			array(
				'label'     => esc_html__( 'پس‌زمینه دکمه (هاور)', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .dwp-add-to-cart-btn:hover' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'button_border_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه‌های دکمه', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 8,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-add-to-cart-btn' => 'border-radius: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'button_padding',
			array(
				'label'      => esc_html__( 'فاصله داخلی دکمه', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-add-to-cart-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'button_height',
			array(
				'label'      => esc_html__( 'ارتفاع دکمه', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 30,
						'max' => 80,
						'step' => 1,
					),
					'em' => array(
						'min' => 1,
						'max' => 3,
						'step' => 0.1,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 38,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-add-to-cart-btn' => 'min-height: {{SIZE}}{{UNIT}}; height: auto;',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل برچسب تخفیف
		$this->start_controls_section(
			'badge_style_section',
			array(
				'label' => esc_html__( 'استایل برچسب تخفیف', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'discount_badge_background',
			array(
				'label'     => esc_html__( 'پس‌زمینه برچسب', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ff6b6b',
				'selectors' => array(
					'{{WRAPPER}} .dwp-discount-badge' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'discount_badge_color',
			array(
				'label'     => esc_html__( 'رنگ متن برچسب', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .dwp-discount-badge' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'badge_typography',
				'label'    => esc_html__( 'تایپوگرافی برچسب', 'disto' ),
				'selector' => '{{WRAPPER}} .dwp-discount-badge',
			)
		);

		$this->end_controls_section();

		// بخش استایل امتیاز
		$this->start_controls_section(
			'rating_style_section',
			array(
				'label' => esc_html__( 'استایل امتیاز', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'rating_color',
			array(
				'label'     => esc_html__( 'رنگ ستاره‌های امتیاز', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffd700',
				'selectors' => array(
					'{{WRAPPER}} .dwp-product-rating .star-rating span' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل دکمه علاقه‌مندی
		$this->start_controls_section(
			'favorite_btn_style_section',
			array(
				'label' => esc_html__( 'استایل دکمه علاقه‌مندی', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'favorite_btn_background',
			array(
				'label'     => esc_html__( 'پس‌زمینه دکمه', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(255, 255, 255, 0.95)',
				'selectors' => array(
					'{{WRAPPER}} .dwp-fav-btn' => 'background: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'favorite_btn_color',
			array(
				'label'     => esc_html__( 'رنگ آیکون', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#9ca3af',
				'selectors' => array(
					'{{WRAPPER}} .dwp-fav-btn' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'favorite_btn_hover_color',
			array(
				'label'     => esc_html__( 'رنگ آیکون (Hover)', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ef4444',
				'selectors' => array(
					'{{WRAPPER}} .dwp-fav-btn:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .dwp-fav-btn.fav' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'favorite_btn_size',
			array(
				'label'      => esc_html__( 'اندازه دکمه', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 30,
						'max' => 60,
						'step' => 1,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 40,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-fav-btn' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// بخش استایل بخش اطلاعات محصول (info section)
		$this->start_controls_section(
			'info_section_style',
			array(
				'label' => esc_html__( 'استایل بخش اطلاعات محصول', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'info_section_padding',
			array(
				'label'      => esc_html__( 'فاصله داخلی بخش اطلاعات', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-info' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'info_section_gap',
			array(
				'label'      => esc_html__( 'فاصله بین المان‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 30,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 12,
				),
				'selectors'  => array(
					'{{WRAPPER}} .dwp-product-info' => 'gap: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		// ساختار query برای محصولات
		$args = array(
			'post_type'      => 'product',
			'posts_per_page' => isset( $settings['products_count'] ) ? absint( $settings['products_count'] ) : 8,
			'post_status'    => 'publish',
		);

		// فیلتر بر اساس دسته‌بندی
		if ( ! empty( $settings['product_category'] ) && $settings['product_category'] > 0 ) {
			$args['tax_query'] = array(
				array(
					'taxonomy' => 'product_cat',
					'field'    => 'term_id',
					'terms'    => (int) $settings['product_category'],
				),
			);
		}

		// مرتب‌سازی
		switch ( $settings['order_by'] ) {
			case 'price':
				$args['meta_key'] = '_price';
				$args['orderby']  = 'meta_value_num';
				$args['order']    = 'ASC';
				break;
			case 'price-desc':
				$args['meta_key'] = '_price';
				$args['orderby']  = 'meta_value_num';
				$args['order']    = 'DESC';
				break;
			case 'popularity':
				$args['meta_key'] = 'total_sales';
				$args['orderby']  = 'meta_value_num';
				$args['order']    = 'DESC';
				break;
			case 'rating':
				$args['meta_key'] = '_wc_average_rating';
				$args['orderby']  = 'meta_value_num';
				$args['order']    = 'DESC';
				break;
			case 'rand':
				$args['orderby'] = 'rand';
				break;
			case 'title':
				$args['orderby'] = 'title';
				$args['order']   = 'ASC';
				break;
			default: // date
				$args['orderby'] = 'date';
				$args['order']   = 'DESC';
				break;
		}

		$query = new \WP_Query( $args );

		if ( ! $query->have_posts() ) {
			echo '<p>' . esc_html__( 'محصولی یافت نشد.', 'disto' ) . '</p>';
			return;
		}

		$widget_id = 'dwp-products-' . $this->get_id();
		
		// دریافت تعداد محصولات در هر ردیف (ریسپانسیو)
		// Elementor responsive controls به صورت products_per_row_tablet و products_per_row_mobile ذخیره می‌شوند
		$products_per_row_desktop = isset( $settings['products_per_row'] ) ? absint( $settings['products_per_row'] ) : 4;
		$products_per_row_tablet  = isset( $settings['products_per_row_tablet'] ) && $settings['products_per_row_tablet'] !== '' ? absint( $settings['products_per_row_tablet'] ) : ( isset( $settings['products_per_row'] ) ? absint( $settings['products_per_row'] ) : 2 );
		$products_per_row_mobile  = isset( $settings['products_per_row_mobile'] ) && $settings['products_per_row_mobile'] !== '' ? absint( $settings['products_per_row_mobile'] ) : 1;
		
		// محدود کردن مقادیر
		$products_per_row_desktop = max( 1, min( 10, $products_per_row_desktop ) );
		$products_per_row_tablet  = max( 1, min( 10, $products_per_row_tablet ) );
		$products_per_row_mobile  = max( 1, min( 10, $products_per_row_mobile ) );

		// بررسی فعال بودن hover effect
		$hover_class = '';
		if ( empty( $settings['enable_hover_effect'] ) || $settings['enable_hover_effect'] !== 'yes' ) {
			$hover_class = 'no-hover-effect';
		}
		?>
		<div id="<?php echo esc_attr( $widget_id ); ?>" class="dwp-products-wrapper" dir="rtl" 
			data-products-per-row-desktop="<?php echo esc_attr( $products_per_row_desktop ); ?>"
			data-products-per-row-tablet="<?php echo esc_attr( $products_per_row_tablet ); ?>"
			data-products-per-row-mobile="<?php echo esc_attr( $products_per_row_mobile ); ?>">
			<div class="dwp-products-grid">
				<?php
				while ( $query->have_posts() ) :
					$query->the_post();
					global $product;

					if ( ! $product ) {
						continue;
					}

					$product_id    = $product->get_id();
					$product_title = $product->get_name();
					$product_url   = $product->get_permalink();
					$product_image = wp_get_attachment_image_url( $product->get_image_id(), 'woocommerce_thumbnail' );
					if ( ! $product_image ) {
						$product_image = wc_placeholder_img_src();
					}

					// محاسبه تخفیف
					$discount_percentage = 0;
					if ( $product->is_on_sale() ) {
						$regular_price = $product->get_regular_price();
						$sale_price    = $product->get_sale_price();
						if ( $regular_price > 0 ) {
							$discount_percentage = round( ( ( $regular_price - $sale_price ) / $regular_price ) * 100 );
						}
					}

					// نمایش امتیاز
					$rating_count = $product->get_rating_count();
					$average      = $product->get_average_rating();
					?>
					<div class="dwp-product-card <?php echo esc_attr( $hover_class ); ?>">
						<!-- Image Container -->
						<div class="dwp-product-image-wrapper">
							<a href="<?php echo esc_url( $product_url ); ?>" class="dwp-product-link" style="padding: 0;">
								<?php if ( $product_image ) : ?>
									<img src="<?php echo esc_url( $product_image ); ?>" alt="<?php echo esc_attr( $product_title ); ?>" class="dwp-product-image" loading="lazy" style="width: 180px; height: 260px;" />
								<?php endif; ?>
							</a>
							
							<!-- دکمه علاقه‌مندی -->
							<?php if ( ! empty( $settings['show_favorite_btn'] ) ) : ?>
								<button type="button" class="dwp-fav-btn" aria-label="<?php echo esc_attr__( 'افزودن به علاقه‌مندی‌ها', 'disto' ); ?>">
									<?php
									if ( ! empty( $settings['favorite_button_icon'] ) ) {
										\Elementor\Icons_Manager::render_icon( $settings['favorite_button_icon'], array( 'aria-hidden' => 'true' ) );
									} else {
										echo '<i class="fas fa-heart"></i>';
									}
									?>
								</button>
							<?php endif; ?>

							<!-- برچسب موجودی -->
							<?php if ( ! empty( $settings['show_stock_badge'] ) ) : ?>
								<?php if ( $product->is_in_stock() ) : ?>
									<span class="dwp-stock-badge"><?php echo esc_html__( 'موجود', 'disto' ); ?></span>
								<?php else : ?>
									<span class="dwp-stock-badge dwp-out-of-stock"><?php echo esc_html__( 'ناموجود', 'disto' ); ?></span>
								<?php endif; ?>
							<?php endif; ?>
							
							<!-- برچسب تخفیف -->
							<?php if ( ! empty( $settings['show_discount_badge'] ) && $discount_percentage > 0 ) : ?>
								<span class="dwp-discount-badge"><?php echo esc_html( '-' . $discount_percentage . '%' ); ?></span>
							<?php endif; ?>

							<!-- امتیاز محصول -->
							<?php if ( ! empty( $settings['show_rating'] ) && $rating_count > 0 ) : ?>
								<div class="dwp-product-rating">
									<?php
									echo wc_get_rating_html( $average ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
									?>
								</div>
							<?php endif; ?>
						</div>

						<!-- بخش اطلاعات محصول -->
						<div class="dwp-product-info">
							<a href="<?php echo esc_url( $product_url ); ?>" class="dwp-product-link">
								<h3 class="dwp-product-title"><?php echo esc_html( $product_title ); ?></h3>
							</a>
							
							<div class="dwp-product-price">
								<?php echo $this->format_price_html( $product->get_price_html() ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
							</div>

							<?php if ( ! empty( $settings['show_add_to_cart'] ) ) : ?>
								<button type="button" class="dwp-add-to-cart-btn" data-product-id="<?php echo esc_attr( $product_id ); ?>">
									<?php
									if ( ! empty( $settings['cart_button_icon'] ) ) {
										\Elementor\Icons_Manager::render_icon( $settings['cart_button_icon'], array( 'aria-hidden' => 'true' ) );
									} else {
										echo '<i class="fas fa-shopping-cart"></i>';
									}
									?>
									<span><?php echo esc_html__( 'خرید', 'disto' ); ?></span>
								</button>
							<?php endif; ?>
						</div>
					</div>
					<?php
				endwhile;
				wp_reset_postdata();
				?>
			</div>
		</div>
		<?php
	}
}
