<?php
/**
 * RTL News Carousel Pro (Elementor Widget)
 *
 * - RTL friendly (Persian sites)
 * - Fetches posts via WP REST API (frontend JS)
 * - Uses Swiper (no jQuery)
 * - Scoped styles with rnc- prefix
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;

class Disto_RTL_News_Carousel_Pro_Widget extends Widget_Base {

	public function get_name() {
		return 'rnc_rtl_news_carousel_pro';
	}

	public function get_title() {
		return esc_html__( 'پست شبکه‌ای', 'disto' );
	}

	public function get_icon() {
		return 'eicon-slider-push';
	}

	public function get_categories() {
		return array( 'disto-elements' );
	}

	public function get_style_depends() {
		// از همان استایل کارت‌های کروسول استفاده می‌کنیم.
		return array( 'disto-child-rnc' );
	}

	public function get_script_depends() {
		// گرید پست شبکه‌ای اسلایدر ندارد، پس نیازی به JS اختصاصی نیست.
		return array();
	}

	private function estimate_reading_time_minutes( $text ) {
		$text  = wp_strip_all_tags( (string) $text );
		$words = preg_split( '/\s+/u', trim( $text ) );
		$count = is_array( $words ) ? count( array_filter( $words ) ) : 0;
		$wpm   = 200;
		return max( 1, (int) round( $count / $wpm ) );
	}

	private function trim_words_fa( $text, $max_words = 28 ) {
		$text = wp_strip_all_tags( (string) $text );
		$text = preg_replace( '/\s+/u', ' ', trim( $text ) );
		if ( '' === $text ) {
			return '';
		}
		$words = preg_split( '/\s+/u', $text );
		if ( ! is_array( $words ) || count( $words ) <= $max_words ) {
			return $text;
		}
		return implode( ' ', array_slice( $words, 0, $max_words ) ) . '…';
	}

	protected function register_controls() {
		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'تنظیمات محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		// Categories options.
		$terms       = get_terms(
			array(
				'taxonomy'   => 'category',
				'hide_empty' => false,
			)
		);
		$cat_options = array(
			0 => esc_html__( 'همه دسته‌ها', 'disto' ),
		);
		if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
			foreach ( $terms as $term ) {
				$cat_options[ (int) $term->term_id ] = $term->name;
			}
		}

		$this->add_control(
			'category_id',
			array(
				'label'   => esc_html__( 'دسته نوشته‌ها', 'disto' ),
				'type'    => Controls_Manager::SELECT2,
				'options' => $cat_options,
				'default' => 0,
			)
		);

		$this->add_control(
			'posts_count',
			array(
				'label'   => esc_html__( 'تعداد نوشته‌ها', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 24,
				'step'    => 1,
				'default' => 9,
			)
		);

		$this->add_control(
			'order_mode',
			array(
				'label'   => esc_html__( 'مرتب‌سازی', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'latest',
				'options' => array(
					'latest'  => esc_html__( 'جدیدترین', 'disto' ),
					'popular' => esc_html__( 'محبوب (بر اساس تعداد دیدگاه)', 'disto' ),
					'random'  => esc_html__( 'تصادفی', 'disto' ),
				),
			)
		);

		$this->add_control(
			'show_excerpt',
			array(
				'label'   => esc_html__( 'نمایش خلاصه', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_date',
			array(
				'label'   => esc_html__( 'نمایش تاریخ', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_category_badge',
			array(
				'label'   => esc_html__( 'نمایش برچسب دسته', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_read_more',
			array(
				'label'   => esc_html__( 'نمایش «ادامه مطلب»', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_reading_time',
			array(
				'label'   => esc_html__( 'نمایش زمان مطالعه', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_autoplay',
			array(
				'label'   => esc_html__( 'اتوماتیک (Autoplay)', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'autoplay_speed',
			array(
				'label'     => esc_html__( 'سرعت autoplay (میلی‌ثانیه)', 'disto' ),
				'type'      => Controls_Manager::NUMBER,
				'min'       => 1000,
				'max'       => 15000,
				'step'      => 250,
				'default'   => 3500,
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->add_control(
			'show_progress',
			array(
				'label'     => esc_html__( 'نمایش نوار پیشرفت autoplay', 'disto' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->add_control(
			'enable_loop',
			array(
				'label'   => esc_html__( 'چرخش (Loop)', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_arrows',
			array(
				'label'   => esc_html__( 'نمایش فلش‌ها', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_dots',
			array(
				'label'   => esc_html__( 'نمایش نقطه‌ها', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_responsive_control(
			'space_between',
			array(
				'label'      => esc_html__( 'فاصله بین اسلایدها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 40,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 16,
				),
			)
		);

		$this->end_controls_section();

		// Style.
		$this->start_controls_section(
			'style_section',
			array(
				'label' => esc_html__( 'استایل', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'card_bg',
			array(
				'label'     => esc_html__( 'رنگ پس‌زمینه کارت', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .rnc-card' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'card_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 40,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 16,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rnc-card'     => 'border-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rnc-card-img' => 'border-radius: {{SIZE}}{{UNIT}};',
				),
			)
		);

		if ( class_exists( '\Elementor\Group_Control_Box_Shadow' ) ) {
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				array(
					'name'     => 'card_shadow',
					'selector' => '{{WRAPPER}} .rnc-card',
				)
			);
		}

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'label'    => esc_html__( 'تایپوگرافی عنوان', 'disto' ),
				'selector' => '{{WRAPPER}} .rnc-title',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'excerpt_typography',
				'label'    => esc_html__( 'تایپوگرافی خلاصه', 'disto' ),
				'selector' => '{{WRAPPER}} .rnc-excerpt',
			)
		);

		$this->add_control(
			'read_more_color',
			array(
				'label'     => esc_html__( 'رنگ «ادامه مطلب»', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-read-more' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'read_more_hover_color',
			array(
				'label'     => esc_html__( 'رنگ هاور «ادامه مطلب»', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-card:hover .rnc-read-more' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'badge_color',
			array(
				'label'     => esc_html__( 'رنگ برچسب دسته', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-badge' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'arrow_color',
			array(
				'label'     => esc_html__( 'رنگ فلش‌ها', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-nav button' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'rtl_padding',
			array(
				'label'      => esc_html__( 'تنظیم فاصله RTL (پدینگ محتوا)', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 32,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 14,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rnc-card-body' => 'padding: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$args = array(
			'post_type'           => 'post',
			'posts_per_page'      => isset( $settings['posts_count'] ) ? absint( $settings['posts_count'] ) : 9,
			'post_status'         => 'publish',
			'ignore_sticky_posts' => true,
			'no_found_rows'       => true,
		);

		if ( ! empty( $settings['category_id'] ) ) {
			$args['cat'] = (int) $settings['category_id'];
		}

		if ( isset( $settings['order_mode'] ) && 'random' === $settings['order_mode'] ) {
			$args['orderby'] = 'rand';
		} elseif ( isset( $settings['order_mode'] ) && 'popular' === $settings['order_mode'] ) {
			$args['orderby'] = 'comment_count';
			$args['order']   = 'DESC';
		} else {
			$args['orderby'] = 'date';
			$args['order']   = 'DESC';
		}

		$query = new \WP_Query( $args );

		if ( ! $query->have_posts() ) {
			return;
		}

		$root_id = 'rnc-grid-' . $this->get_id();
		?>
		<div id="<?php echo esc_attr( $root_id ); ?>" class="rnc-root rnc-root-grid" dir="rtl">
			<div class="rnc-grid">
			<?php
			while ( $query->have_posts() ) :
				$query->the_post();
				$post_id = get_the_ID();
				$title   = get_the_title();
				$link    = get_permalink();

				$thumb_url = get_the_post_thumbnail_url( $post_id, 'large' );
				$content   = get_post_field( 'post_content', $post_id );

				// خلاصه همیشه بر اساس محتوای واقعی پست ساخته می‌شود (مانند ویجت کروسول اخبار)
				$excerpt = '';
				if ( ! empty( $settings['show_excerpt'] ) ) {
					$excerpt = $this->trim_words_fa( $content, 28 );
				}

				$reading_min = 0;
				if ( ! empty( $settings['show_reading_time'] ) ) {
					$reading_min = $this->estimate_reading_time_minutes( $content );
				}

				?>
				<div class="rnc-card">
					<a href="<?php echo esc_url( $link ); ?>">
						<div class="rnc-card-img">
							<?php
							if ( ! empty( $settings['show_category_badge'] ) ) {
								$cats = get_the_category( $post_id );
								if ( ! empty( $cats ) && ! empty( $cats[0]->name ) ) {
									?>
									<span class="rnc-badge"><?php echo esc_html( $cats[0]->name ); ?></span>
									<?php
								}
							}
							?>
							<span class="rnc-overlay" aria-hidden="true"></span>
							<?php if ( $thumb_url ) : ?>
								<img loading="lazy" decoding="async" src="<?php echo esc_url( $thumb_url ); ?>" alt="<?php echo esc_attr( $title ); ?>" />
							<?php endif; ?>
						</div>
						<div class="rnc-card-body">
							<h3 class="rnc-title"><?php echo esc_html( $title ); ?></h3>
							<div class="rnc-meta">
								<span>
									<?php
									if ( ! empty( $settings['show_date'] ) ) {
										echo esc_html( get_the_date( 'j F Y', $post_id ) );
									}
									?>
								</span>
								<span>
									<?php
									if ( ! empty( $settings['show_reading_time'] ) && $reading_min ) {
										echo esc_html( sprintf( _n( '%d دقیقه مطالعه', '%d دقیقه مطالعه', $reading_min, 'disto' ), $reading_min ) );
									}
									?>
								</span>
							</div>
							<?php if ( ! empty( $settings['show_excerpt'] ) && $excerpt ) : ?>
								<p class="rnc-excerpt"><?php echo esc_html( $excerpt ); ?></p>
							<?php endif; ?>
							<?php if ( ! empty( $settings['show_read_more'] ) ) : ?>
								<span class="rnc-read-more"><?php echo esc_html__( 'ادامه مطلب', 'disto' ); ?></span>
							<?php endif; ?>
						</div>
					</a>
				</div>
				<?php
			endwhile;
			wp_reset_postdata();
			?>
			</div>
		</div>
		<?php
	}
}


