<?php
/**
 * RTL News Carousel Cards (Elementor Widget)
 *
 * نسخه کارت‌محور از اسلایدر اخبار، با تنظیمات پویا برای دسته‌بندی و تعداد پست‌ها.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;

class Disto_RTL_News_Carousel_Cards_Widget extends Widget_Base {

	public function get_name() {
		return 'rnc_news_carousel_cards';
	}

	public function get_title() {
		return esc_html__( 'کروسول اخبار', 'disto' );
	}

	public function get_icon() {
		return 'eicon-posts-carousel';
	}

	public function get_categories() {
		return array( 'disto-elements' );
	}

	public function get_style_depends() {
		return array( 'disto-child-swiper', 'disto-child-rnc' );
	}

	public function get_script_depends() {
		return array( 'disto-child-swiper', 'disto-child-rnc' );
	}

	protected function register_controls() {
		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'تنظیمات محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		// Categories options.
		$terms       = get_terms(
			array(
				'taxonomy'   => 'category',
				'hide_empty' => false,
			)
		);
		$cat_options = array(
			0 => esc_html__( 'همه دسته‌ها', 'disto' ),
		);
		if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
			foreach ( $terms as $term ) {
				$cat_options[ (int) $term->term_id ] = $term->name;
			}
		}

		$this->add_control(
			'category_id',
			array(
				'label'   => esc_html__( 'دسته نوشته‌ها', 'disto' ),
				'type'    => Controls_Manager::SELECT2,
				'options' => $cat_options,
				'default' => 0,
			)
		);

		$this->add_control(
			'posts_count',
			array(
				'label'   => esc_html__( 'تعداد نوشته‌ها', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 24,
				'step'    => 1,
				'default' => 9,
			)
		);

		$this->add_control(
			'order_mode',
			array(
				'label'   => esc_html__( 'مرتب‌سازی', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'latest',
				'options' => array(
					'latest'  => esc_html__( 'جدیدترین', 'disto' ),
					'popular' => esc_html__( 'محبوب (بر اساس تعداد دیدگاه)', 'disto' ),
					'random'  => esc_html__( 'تصادفی', 'disto' ),
				),
			)
		);

		$this->add_control(
			'show_excerpt',
			array(
				'label'   => esc_html__( 'نمایش خلاصه', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_date',
			array(
				'label'   => esc_html__( 'نمایش تاریخ', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_category_badge',
			array(
				'label'   => esc_html__( 'نمایش برچسب دسته', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_read_more',
			array(
				'label'   => esc_html__( 'نمایش «ادامه مطلب»', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_reading_time',
			array(
				'label'   => esc_html__( 'نمایش زمان مطالعه', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_autoplay',
			array(
				'label'   => esc_html__( 'اتوماتیک (Autoplay)', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'autoplay_speed',
			array(
				'label'     => esc_html__( 'سرعت autoplay (میلی‌ثانیه)', 'disto' ),
				'type'      => Controls_Manager::NUMBER,
				'min'       => 1000,
				'max'       => 15000,
				'step'      => 250,
				'default'   => 3500,
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->add_control(
			'show_progress',
			array(
				'label'     => esc_html__( 'نمایش نوار پیشرفت autoplay', 'disto' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->add_control(
			'enable_loop',
			array(
				'label'   => esc_html__( 'چرخش (Loop)', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_arrows',
			array(
				'label'   => esc_html__( 'نمایش فلش‌ها', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_dots',
			array(
				'label'   => esc_html__( 'نمایش نقطه‌ها', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_responsive_control(
			'space_between',
			array(
				'label'      => esc_html__( 'فاصله بین اسلایدها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 40,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 16,
				),
			)
		);

		$this->end_controls_section();

		// Style.
		$this->start_controls_section(
			'style_section',
			array(
				'label' => esc_html__( 'استایل کارت‌ها', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'card_bg',
			array(
				'label'     => esc_html__( 'رنگ پس‌زمینه کارت', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .rnc-card' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'card_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 40,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 16,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rnc-card'     => 'border-radius: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rnc-card-img' => 'border-radius: {{SIZE}}{{UNIT}};',
				),
			)
		);

		if ( class_exists( '\Elementor\Group_Control_Box_Shadow' ) ) {
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				array(
					'name'     => 'card_shadow',
					'selector' => '{{WRAPPER}} .rnc-card',
				)
			);
		}

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'label'    => esc_html__( 'تایپوگرافی عنوان', 'disto' ),
				'selector' => '{{WRAPPER}} .rnc-title',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'excerpt_typography',
				'label'    => esc_html__( 'تایپوگرافی خلاصه', 'disto' ),
				'selector' => '{{WRAPPER}} .rnc-excerpt',
			)
		);

		$this->add_control(
			'read_more_color',
			array(
				'label'     => esc_html__( 'رنگ «ادامه مطلب»', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-read-more' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'read_more_hover_color',
			array(
				'label'     => esc_html__( 'رنگ هاور «ادامه مطلب»', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-card:hover .rnc-read-more' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'badge_color',
			array(
				'label'     => esc_html__( 'رنگ برچسب دسته', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-badge' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'arrow_color',
			array(
				'label'     => esc_html__( 'رنگ فلش‌ها', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rnc-nav button' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'rtl_padding',
			array(
				'label'      => esc_html__( 'پدینگ محتوا', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 32,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 14,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rnc-card-body' => 'padding: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$payload = array(
			'categoryId'      => isset( $settings['category_id'] ) ? (int) $settings['category_id'] : 0,
			'postsCount'      => isset( $settings['posts_count'] ) ? absint( $settings['posts_count'] ) : 9,
			'orderMode'       => isset( $settings['order_mode'] ) ? sanitize_text_field( $settings['order_mode'] ) : 'latest',
			'showExcerpt'     => ! empty( $settings['show_excerpt'] ),
			'showDate'        => ! empty( $settings['show_date'] ),
			'showBadge'       => ! empty( $settings['show_category_badge'] ),
			'showReadMore'    => ! empty( $settings['show_read_more'] ),
			'showReadingTime' => ! empty( $settings['show_reading_time'] ),
			'autoplay'        => ! empty( $settings['enable_autoplay'] ),
			'autoplaySpeed'   => isset( $settings['autoplay_speed'] ) ? absint( $settings['autoplay_speed'] ) : 3500,
			'showProgress'    => ! empty( $settings['show_progress'] ),
			'loop'            => ! empty( $settings['enable_loop'] ),
			'arrows'          => ! empty( $settings['enable_arrows'] ),
			'dots'            => ! empty( $settings['enable_dots'] ),
			'spaceBetween'    => isset( $settings['space_between']['size'] ) ? (int) $settings['space_between']['size'] : 16,
		);

		if ( $payload['postsCount'] < 1 ) {
			$payload['postsCount'] = 1;
		}
		if ( $payload['postsCount'] > 24 ) {
			$payload['postsCount'] = 24;
		}
		if ( $payload['autoplaySpeed'] < 1000 ) {
			$payload['autoplaySpeed'] = 1000;
		}

		$root_id = 'rnc-cards-' . $this->get_id();
		?>
		<div id="<?php echo esc_attr( $root_id ); ?>" class="rnc-root" dir="rtl" data-rnc-settings="<?php echo esc_attr( wp_json_encode( $payload ) ); ?>">
			<div class="rnc-header">
				<?php if ( $payload['arrows'] ) : ?>
					<div class="rnc-nav" aria-label="<?php echo esc_attr__( 'Navigation', 'disto' ); ?>">
						<button type="button" class="rnc-prev" aria-label="<?php echo esc_attr__( 'Previous', 'disto' ); ?>">&lsaquo;</button>
						<button type="button" class="rnc-next" aria-label="<?php echo esc_attr__( 'Next', 'disto' ); ?>">&rsaquo;</button>
					</div>
				<?php endif; ?>
				<?php if ( $payload['showProgress'] && $payload['autoplay'] ) : ?>
					<div class="rnc-progress" aria-hidden="true"><span class="rnc-progress-bar"></span></div>
				<?php endif; ?>
			</div>

			<div class="swiper rnc-swiper">
				<div class="swiper-wrapper">
					<?php for ( $i = 0; $i < 3; $i++ ) : ?>
						<div class="swiper-slide">
							<div class="rnc-card rnc-skeleton" aria-hidden="true">
								<div class="rnc-card-img"></div>
								<div class="rnc-card-body">
									<div class="rnc-skeleton-line rnc-skeleton-title"></div>
									<div class="rnc-skeleton-line"></div>
									<div class="rnc-skeleton-line rnc-skeleton-short"></div>
								</div>
							</div>
						</div>
					<?php endfor; ?>
				</div>
				<?php if ( $payload['dots'] ) : ?>
					<div class="swiper-pagination rnc-pagination"></div>
				<?php endif; ?>
			</div>
		</div>
		<?php
	}
}


