<?php
/**
 * Pro Portfolio Widget (Elementor)
 *
 * RTL-friendly portfolio widget with 3 skins: Azure Avenue, Creative Agency, Modern Bento.
 * Supports Swiper, Masonry, AJAX filtering, lazy load, lightbox, entrance animations.
 *
 * @package Disto_Child
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;

class Disto_Pro_Portfolio_Widget extends Widget_Base {

	public function get_name() {
		return 'disto_pro_portfolio';
	}

	public function get_title() {
		return esc_html__( 'Pro Portfolio', 'disto' );
	}

	public function get_icon() {
		return 'eicon-gallery-grid';
	}

	public function get_categories() {
		return array( 'disto-elements', 'general' );
	}

	public function get_style_depends() {
		return array( 'disto-child-swiper', 'disto-child-pro-portfolio' );
	}

	public function get_script_depends() {
		return array( 'disto-child-swiper', 'disto-child-pro-portfolio' );
	}

	protected function register_controls() {
		$this->register_skin_content_controls();
		$this->register_content_source_controls();
		$this->register_repeater_controls();
		$this->register_query_controls();
		$this->register_style1_controls();
		$this->register_style2_controls();
		$this->register_style3_controls();
		$this->register_style_controls();
	}

	private function register_skin_content_controls() {
		$this->start_controls_section(
			'section_skin',
			array(
				'label' => esc_html__( 'Skin', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'skin',
			array(
				'label'   => esc_html__( 'Skin', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'azure',
				'options' => array(
					'azure'   => esc_html__( 'Style 1 – Azure Avenue', 'disto' ),
					'creative' => esc_html__( 'Style 2 – Creative Agency', 'disto' ),
					'bento'   => esc_html__( 'Style 3 – Modern Bento', 'disto' ),
				),
			)
		);

		$this->end_controls_section();
	}

	private function register_content_source_controls() {
		$this->start_controls_section(
			'section_content_source',
			array(
				'label' => esc_html__( 'Content Source', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'content_source',
			array(
				'label'   => esc_html__( 'Source', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'manual',
				'options' => array(
					'manual' => esc_html__( 'Manual (Repeater)', 'disto' ),
					'query'  => esc_html__( 'Post Type Query', 'disto' ),
				),
			)
		);

		$this->end_controls_section();
	}

	private function register_repeater_controls() {
		$this->start_controls_section(
			'section_items',
			array(
				'label'     => esc_html__( 'Portfolio Items', 'disto' ),
				'tab'       => Controls_Manager::TAB_CONTENT,
				'condition' => array( 'content_source' => 'manual' ),
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'image',
			array(
				'label'   => esc_html__( 'Image', 'disto' ),
				'type'    => Controls_Manager::MEDIA,
				'default' => array(),
			)
		);

		$repeater->add_control(
			'title',
			array(
				'label'       => esc_html__( 'Title', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Project Title', 'disto' ),
				'placeholder' => esc_html__( 'Project Title', 'disto' ),
			)
		);

		$repeater->add_control(
			'description',
			array(
				'label'       => esc_html__( 'Description', 'disto' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => '',
				'placeholder' => esc_html__( 'Project description…', 'disto' ),
			)
		);

		$repeater->add_control(
			'client_name',
			array(
				'label'       => esc_html__( 'Client Name', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'placeholder' => esc_html__( 'Client', 'disto' ),
			)
		);

		$repeater->add_control(
			'date',
			array(
				'label'       => esc_html__( 'Date', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'placeholder' => esc_html__( 'Jul 27 – 2024', 'disto' ),
			)
		);

		$repeater->add_control(
			'category',
			array(
				'label'       => esc_html__( 'Category', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'placeholder' => esc_html__( 'Branding', 'disto' ),
			)
		);

		$repeater->add_control(
			'custom_tags',
			array(
				'label'       => esc_html__( 'Custom Tags', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'placeholder' => esc_html__( 'CONSULTING, ENGINEERING, RENOVATION', 'disto' ),
				'description' => esc_html__( 'Comma-separated tags', 'disto' ),
			)
		);

		$repeater->add_control(
			'external_link',
			array(
				'label'         => esc_html__( 'External Link', 'disto' ),
				'type'          => Controls_Manager::URL,
				'placeholder'   => 'https://',
				'show_external' => true,
				'default'       => array(
					'url'         => '',
					'is_external' => false,
					'nofollow'    => false,
				),
			)
		);

		$repeater->add_control(
			'lightbox_image',
			array(
				'label'   => esc_html__( 'Lightbox Image / Video', 'disto' ),
				'type'    => Controls_Manager::MEDIA,
				'default' => array(),
			)
		);

		$repeater->add_control(
			'lightbox_video_url',
			array(
				'label'       => esc_html__( 'Or Lightbox Video URL', 'disto' ),
				'type'        => Controls_Manager::URL,
				'placeholder' => 'https://www.youtube.com/watch?v=',
			)
		);

		$repeater->add_control(
			'button_text',
			array(
				'label'       => esc_html__( 'Button Text', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'placeholder' => esc_html__( 'View Project', 'disto' ),
			)
		);

		$repeater->add_control(
			'custom_bg_color',
			array(
				'label' => esc_html__( 'Custom Background Color', 'disto' ),
				'type'  => Controls_Manager::COLOR,
			)
		);

		$repeater->add_control(
			'custom_accent_color',
			array(
				'label' => esc_html__( 'Custom Accent Color', 'disto' ),
				'type'  => Controls_Manager::COLOR,
			)
		);

		$repeater->add_control(
			'author_name',
			array(
				'label'       => esc_html__( 'Author', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => '',
				'placeholder' => esc_html__( 'Author Name', 'disto' ),
			)
		);

		$this->add_control(
			'portfolio_items',
			array(
				'label'       => esc_html__( 'Items', 'disto' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => array(
					array(
						'title'    => esc_html__( 'Azure Avenue', 'disto' ),
						'category' => esc_html__( 'Branding', 'disto' ),
						'date'     => esc_html__( 'Jul 27 – 2024', 'disto' ),
						'client_name' => esc_html__( 'Section Express', 'disto' ),
						'author_name' => esc_html__( 'Kevin Anderson', 'disto' ),
						'custom_tags' => esc_html__( 'CONSULTING, ENGINEERING, RENOVATION', 'disto' ),
					),
					array(
						'title'    => esc_html__( 'Project Two', 'disto' ),
						'category' => esc_html__( 'Web Design', 'disto' ),
					),
					array(
						'title'    => esc_html__( 'Project Three', 'disto' ),
						'category' => esc_html__( 'Consulting', 'disto' ),
					),
				),
				'title_field' => '{{{ title }}}',
			)
		);

		$this->end_controls_section();
	}

	private function register_query_controls() {
		$this->start_controls_section(
			'section_query',
			array(
				'label'     => esc_html__( 'Query', 'disto' ),
				'tab'       => Controls_Manager::TAB_CONTENT,
				'condition' => array( 'content_source' => 'query' ),
			)
		);

		$post_types = get_post_types( array( 'public' => true ), 'objects' );
		$pt_options = array();
		foreach ( $post_types as $pt ) {
			$pt_options[ $pt->name ] = $pt->label;
		}

		$this->add_control(
			'post_type',
			array(
				'label'   => esc_html__( 'Post Type', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'post',
				'options' => $pt_options,
			)
		);

		$taxonomies = get_taxonomies( array( 'public' => true ), 'objects' );
		$tax_options = array( '' => esc_html__( 'All', 'disto' ) );
		foreach ( $taxonomies as $tax ) {
			$tax_options[ $tax->name ] = $tax->label;
		}

		$this->add_control(
			'taxonomy_filter',
			array(
				'label'   => esc_html__( 'Category Taxonomy', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'category',
				'options' => $tax_options,
			)
		);

		$this->add_control(
			'posts_count',
			array(
				'label'   => esc_html__( 'Posts Count', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 50,
				'default' => 9,
			)
		);

		$this->add_control(
			'orderby',
			array(
				'label'   => esc_html__( 'Order By', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => array(
					'date'  => esc_html__( 'Date', 'disto' ),
					'title' => esc_html__( 'Title', 'disto' ),
					'rand'  => esc_html__( 'Random', 'disto' ),
				),
			)
		);

		$this->end_controls_section();
	}

	private function register_style1_controls() {
		$this->start_controls_section(
			'section_style1',
			array(
				'label'     => esc_html__( 'Azure Avenue Options', 'disto' ),
				'tab'       => Controls_Manager::TAB_CONTENT,
				'condition' => array( 'skin' => 'azure' ),
			)
		);

		$this->add_control(
			'azure_autoplay',
			array(
				'label'   => esc_html__( 'Autoplay', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'azure_loop',
			array(
				'label'   => esc_html__( 'Loop', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'azure_arrows',
			array(
				'label'   => esc_html__( 'Navigation Arrows', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'azure_pagination',
			array(
				'label'   => esc_html__( 'Pagination', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'no',
			)
		);

		$this->add_control(
			'azure_meta_follow_slide',
			array(
				'label'   => esc_html__( 'Meta Follows Active Slide', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'azure_show_share',
			array(
				'label'   => esc_html__( 'Share Buttons', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();
	}

	private function register_style2_controls() {
		$this->start_controls_section(
			'section_style2',
			array(
				'label'     => esc_html__( 'Creative Agency Options', 'disto' ),
				'tab'       => Controls_Manager::TAB_CONTENT,
				'condition' => array( 'skin' => 'creative' ),
			)
		);

		$this->add_control(
			'creative_title_html',
			array(
				'label'       => esc_html__( 'Section Title (HTML allowed)', 'disto' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => esc_html__( 'Work', 'disto' ),
				'placeholder' => esc_html__( 'Work →', 'disto' ),
			)
		);

		$this->add_control(
			'creative_subtitle',
			array(
				'label'       => esc_html__( 'Subtitle / Tagline', 'disto' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => '',
				'placeholder' => esc_html__( 'We provide brilliant idea to grow the startup – agency with your sharp brand.', 'disto' ),
			)
		);

		$this->add_control(
			'creative_columns',
			array(
				'label'   => esc_html__( 'Columns', 'disto' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'px' => array( 'min' => 2, 'max' => 6 ),
				),
				'default' => array( 'size' => 4 ),
			)
		);

		$this->add_control(
			'creative_show_overlay',
			array(
				'label'   => esc_html__( 'Show Overlay', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'creative_hover_zoom',
			array(
				'label'   => esc_html__( 'Hover Zoom Effect', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();
	}

	private function register_style3_controls() {
		$this->start_controls_section(
			'section_style3',
			array(
				'label'     => esc_html__( 'Modern Bento Options', 'disto' ),
				'tab'       => Controls_Manager::TAB_CONTENT,
				'condition' => array( 'skin' => 'bento' ),
			)
		);

		$this->add_control(
			'bento_masonry',
			array(
				'label'   => esc_html__( 'Masonry Layout', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'bento_border_radius',
			array(
				'label'   => esc_html__( 'Border Radius (px)', 'disto' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'px' => array( 'min' => 0, 'max' => 80 ),
				),
				'default' => array( 'size' => 24 ),
			)
		);

		$this->add_control(
			'bento_gap',
			array(
				'label'   => esc_html__( 'Gap (px)', 'disto' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'px' => array( 'min' => 0, 'max' => 60 ),
				),
				'default' => array( 'size' => 16 ),
			)
		);

		$this->add_control(
			'bento_hover_animation',
			array(
				'label'   => esc_html__( 'Hover Animation', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'reveal',
				'options' => array(
					'reveal' => esc_html__( 'Reveal Content', 'disto' ),
					'zoom'   => esc_html__( 'Zoom In Image', 'disto' ),
					'both'   => esc_html__( 'Both', 'disto' ),
				),
			)
		);

		$this->add_control(
			'bento_yellow_accent',
			array(
				'label'   => esc_html__( 'Yellow Accent', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'no',
			)
		);

		$this->end_controls_section();
	}

	private function register_style_controls() {
		$this->start_controls_section(
			'section_style_general',
			array(
				'label' => esc_html__( 'General Style', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'accent_color',
			array(
				'label'     => esc_html__( 'Accent Color', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#7c3aed',
				'selectors' => array(
					'{{WRAPPER}} .ppw-accent' => 'color: {{VALUE}};',
					'{{WRAPPER}} .ppw-meta-label' => 'color: {{VALUE}};',
					'{{WRAPPER}} .ppw-tag-badge' => 'border-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'overlay_bg_color',
			array(
				'label'     => esc_html__( 'Overlay Background', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0,0,0,0.6)',
				'selectors' => array(
					'{{WRAPPER}} .ppw-overlay' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'hover_overlay_color',
			array(
				'label'     => esc_html__( 'Hover Overlay Color', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0,0,0,0.5)',
				'selectors' => array(
					'{{WRAPPER}} .ppw-item:hover .ppw-overlay' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'image_zoom_intensity',
			array(
				'label'   => esc_html__( 'Image Zoom Intensity (%)', 'disto' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'px' => array( 'min' => 100, 'max' => 130 ),
				),
				'default' => array( 'size' => 110 ),
				'selectors' => array(
					'{{WRAPPER}}' => '--ppw-zoom: {{SIZE}};',
				),
			)
		);

		$this->add_control(
			'grid_gap',
			array(
				'label'   => esc_html__( 'Grid Gap (px)', 'disto' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'px' => array( 'min' => 0, 'max' => 60 ),
				),
				'default' => array( 'size' => 16 ),
				'selectors' => array(
					'{{WRAPPER}}.ppw-skin-creative .ppw-grid' => 'gap: {{SIZE}}px;',
				),
			)
		);

		$this->add_control(
			'border_radius_global',
			array(
				'label'   => esc_html__( 'Border Radius (px)', 'disto' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'px' => array( 'min' => 0, 'max' => 50 ),
				),
				'default' => array( 'size' => 12 ),
				'selectors' => array(
					'{{WRAPPER}} .ppw-item, {{WRAPPER}} .ppw-img-wrap' => 'border-radius: {{SIZE}}px;',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'box_shadow',
				'label'    => esc_html__( 'Box Shadow', 'disto' ),
				'selector' => '{{WRAPPER}} .ppw-item',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_typography_title',
			array(
				'label' => esc_html__( 'Title Typography', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_title',
				'label'    => esc_html__( 'Title', 'disto' ),
				'selector' => '{{WRAPPER}} .ppw-title',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_typography_desc',
			array(
				'label' => esc_html__( 'Description Typography', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_description',
				'label'    => esc_html__( 'Description', 'disto' ),
				'selector' => '{{WRAPPER}} .ppw-desc',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_typography_meta',
			array(
				'label' => esc_html__( 'Meta Labels Typography', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_meta',
				'label'    => esc_html__( 'Meta Labels', 'disto' ),
				'selector' => '{{WRAPPER}} .ppw-meta-label',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_typography_tags',
			array(
				'label' => esc_html__( 'Tags Typography', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_tags',
				'label'    => esc_html__( 'Tags', 'disto' ),
				'selector' => '{{WRAPPER}} .ppw-tag-badge, {{WRAPPER}} .ppw-category-tag',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_typography_button',
			array(
				'label' => esc_html__( 'Button Typography', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_button',
				'label'    => esc_html__( 'Button', 'disto' ),
				'selector' => '{{WRAPPER}} .ppw-btn',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_entrance_animation',
			array(
				'label' => esc_html__( 'Entrance Animation', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'entrance_animation',
			array(
				'label'   => esc_html__( 'Animation', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'fadeUp',
				'options' => array(
					'none'     => esc_html__( 'None', 'disto' ),
					'fadeUp'   => esc_html__( 'Fade Up', 'disto' ),
					'fadeIn'   => esc_html__( 'Fade In', 'disto' ),
					'slideIn'  => esc_html__( 'Slide In', 'disto' ),
					'zoomIn'   => esc_html__( 'Zoom In', 'disto' ),
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Get portfolio items (from repeater or query).
	 *
	 * @return array
	 */
	private function get_portfolio_items() {
		$settings = $this->get_settings_for_display();

		if ( 'query' === $settings['content_source'] ) {
			return $this->get_items_from_query();
		}

		$items = isset( $settings['portfolio_items'] ) && is_array( $settings['portfolio_items'] ) ? $settings['portfolio_items'] : array();
		$out   = array();

		foreach ( $items as $item ) {
			$out[] = array(
				'image'        => isset( $item['image'] ) ? $item['image'] : array(),
				'title'        => isset( $item['title'] ) ? $item['title'] : '',
				'description'  => isset( $item['description'] ) ? $item['description'] : '',
				'client_name'  => isset( $item['client_name'] ) ? $item['client_name'] : '',
				'date'         => isset( $item['date'] ) ? $item['date'] : '',
				'category'     => isset( $item['category'] ) ? $item['category'] : '',
				'custom_tags'  => isset( $item['custom_tags'] ) ? $item['custom_tags'] : '',
				'external_link'=> isset( $item['external_link'] ) ? $item['external_link'] : array(),
				'lightbox_image' => isset( $item['lightbox_image'] ) ? $item['lightbox_image'] : array(),
				'lightbox_video_url' => isset( $item['lightbox_video_url']['url'] ) ? $item['lightbox_video_url']['url'] : '',
				'button_text'  => isset( $item['button_text'] ) ? $item['button_text'] : '',
				'custom_bg_color' => isset( $item['custom_bg_color'] ) ? $item['custom_bg_color'] : '',
				'custom_accent_color' => isset( $item['custom_accent_color'] ) ? $item['custom_accent_color'] : '',
				'author_name'   => isset( $item['author_name'] ) ? $item['author_name'] : '',
			);
		}

		return $out;
	}

	/**
	 * Get items from WP_Query.
	 *
	 * @return array
	 */
	private function get_items_from_query() {
		$settings = $this->get_settings_for_display();
		$args     = array(
			'post_type'      => $settings['post_type'],
			'posts_per_page' => isset( $settings['posts_count'] ) ? (int) $settings['posts_count'] : 9,
			'orderby'        => isset( $settings['orderby'] ) ? $settings['orderby'] : 'date',
			'order'          => 'DESC',
			'post_status'    => 'publish',
		);

		$tax = isset( $settings['taxonomy_filter'] ) ? $settings['taxonomy_filter'] : '';
		if ( $tax ) {
			$args['tax_query'] = array(); // Optional: add term filter via another control if needed.
		}

		$query = new \WP_Query( $args );
		$out   = array();

		while ( $query->have_posts() ) {
			$query->the_post();
			$thumb = get_the_post_thumbnail_url( get_the_ID(), 'large' );
			$out[] = array(
				'image'             => $thumb ? array( 'url' => $thumb, 'id' => get_post_thumbnail_id() ) : array(),
				'title'             => get_the_title(),
				'description'       => get_the_excerpt(),
				'client_name'       => '',
				'date'              => get_the_date(),
				'category'          => '',
				'custom_tags'       => '',
				'external_link'     => array( 'url' => get_permalink() ),
				'lightbox_image'    => array(),
				'lightbox_video_url'=> '',
				'button_text'       => '',
				'custom_bg_color'   => '',
				'custom_accent_color' => '',
				'author_name'       => get_the_author(),
			);
		}

		wp_reset_postdata();
		return $out;
	}

	protected function render() {
		$settings = $this->get_settings_for_display();
		$items    = $this->get_portfolio_items();
		$skin     = isset( $settings['skin'] ) ? $settings['skin'] : 'azure';

		if ( empty( $items ) ) {
			echo '<p class="ppw-empty">' . esc_html__( 'Add portfolio items in the widget settings.', 'disto' ) . '</p>';
			return;
		}

		$wrapper_class = 'ppw-wrapper ppw-rtl ppw-skin-' . esc_attr( $skin );
		$entrance      = isset( $settings['entrance_animation'] ) ? $settings['entrance_animation'] : 'fadeUp';
		if ( 'none' !== $entrance ) {
			$wrapper_class .= ' ppw-entrance-' . esc_attr( $entrance );
		}
		if ( 'bento' === $skin && ! empty( $settings['bento_yellow_accent'] ) && $settings['bento_yellow_accent'] === 'yes' ) {
			$wrapper_class .= ' ppw-yellow-accent';
		}

		$this->add_render_attribute( 'wrapper', array(
			'class'             => $wrapper_class,
			'data-skin'         => $skin,
			'data-content-source' => $settings['content_source'],
		) );

		if ( 'azure' === $skin ) {
			$this->add_render_attribute( 'wrapper', 'data-autoplay', $settings['azure_autoplay'] === 'yes' ? '1' : '0' );
			$this->add_render_attribute( 'wrapper', 'data-loop', $settings['azure_loop'] === 'yes' ? '1' : '0' );
			$this->add_render_attribute( 'wrapper', 'data-arrows', $settings['azure_arrows'] === 'yes' ? '1' : '0' );
			$this->add_render_attribute( 'wrapper', 'data-pagination', $settings['azure_pagination'] === 'yes' ? '1' : '0' );
			$this->add_render_attribute( 'wrapper', 'data-meta-follow', ! empty( $settings['azure_meta_follow_slide'] ) && $settings['azure_meta_follow_slide'] === 'yes' ? '1' : '0' );
		}

		if ( 'creative' === $skin ) {
			$cols = isset( $settings['creative_columns']['size'] ) ? (int) $settings['creative_columns']['size'] : 4;
			$this->add_render_attribute( 'wrapper', 'data-columns', $cols );
			$this->add_render_attribute( 'wrapper', 'data-overlay', $settings['creative_show_overlay'] === 'yes' ? '1' : '0' );
			$this->add_render_attribute( 'wrapper', 'data-hover-zoom', $settings['creative_hover_zoom'] === 'yes' ? '1' : '0' );
		}

		if ( 'bento' === $skin ) {
			$this->add_render_attribute( 'wrapper', 'data-masonry', $settings['bento_masonry'] === 'yes' ? '1' : '0' );
			$this->add_render_attribute( 'wrapper', 'data-radius', isset( $settings['bento_border_radius']['size'] ) ? (int) $settings['bento_border_radius']['size'] : 24 );
			$this->add_render_attribute( 'wrapper', 'data-gap', isset( $settings['bento_gap']['size'] ) ? (int) $settings['bento_gap']['size'] : 16 );
			$this->add_render_attribute( 'wrapper', 'data-hover-animation', isset( $settings['bento_hover_animation'] ) ? $settings['bento_hover_animation'] : 'reveal' );
		}
		?>
		<div <?php $this->print_render_attribute_string( 'wrapper' ); ?>>
			<?php
			if ( 'azure' === $skin ) {
				$this->render_skin_azure( $items, $settings );
			} elseif ( 'creative' === $skin ) {
				$this->render_skin_creative( $items, $settings );
			} else {
				$this->render_skin_bento( $items, $settings );
			}
			?>
		</div>
		<?php
	}

	private function render_skin_azure( $items, $settings ) {
		$show_share = ! empty( $settings['azure_show_share'] ) && $settings['azure_show_share'] === 'yes';
		?>
		<div class="ppw-azure">
			<div class="ppw-azure-swiper swiper">
				<div class="swiper-wrapper">
					<?php foreach ( $items as $index => $item ) : ?>
						<div class="swiper-slide" data-index="<?php echo (int) $index; ?>">
							<?php $this->render_item_image( $item, $index, $settings ); ?>
						</div>
					<?php endforeach; ?>
				</div>
				<?php if ( ! empty( $settings['azure_arrows'] ) && $settings['azure_arrows'] === 'yes' ) : ?>
					<div class="swiper-button-prev" aria-label="<?php esc_attr_e( 'Previous', 'disto' ); ?>"></div>
					<div class="swiper-button-next" aria-label="<?php esc_attr_e( 'Next', 'disto' ); ?>"></div>
				<?php endif; ?>
				<?php if ( ! empty( $settings['azure_pagination'] ) && $settings['azure_pagination'] === 'yes' ) : ?>
					<div class="swiper-pagination"></div>
				<?php endif; ?>
			</div>
			<div class="ppw-azure-meta">
				<div class="ppw-azure-meta-inner">
					<div class="ppw-azure-left">
						<h2 class="ppw-title ppw-azure-title"><?php echo esc_html( $items[0]['title'] ); ?></h2>
						<?php if ( ! empty( $items[0]['description'] ) ) : ?>
							<div class="ppw-desc ppw-azure-desc"><?php echo wp_kses_post( $items[0]['description'] ); ?></div>
						<?php endif; ?>
						<?php if ( ! empty( $items[0]['custom_tags'] ) ) : ?>
							<div class="ppw-tags-wrap">
								<span class="ppw-tag-badge"><?php echo esc_html( $items[0]['custom_tags'] ); ?></span>
							</div>
						<?php endif; ?>
					</div>
					<div class="ppw-azure-right">
						<?php $this->render_meta_row( 'category', esc_html__( 'Category', 'disto' ), $items[0]['category'] ); ?>
						<?php $this->render_meta_row( 'date', esc_html__( 'Date', 'disto' ), $items[0]['date'] ); ?>
						<?php if ( $show_share ) : ?>
							<div class="ppw-meta-row" data-meta-key="share">
								<span class="ppw-meta-label"><?php esc_html_e( 'Share', 'disto' ); ?></span>
								<div class="ppw-share-btns">
									<a href="#" aria-label="Facebook"><span class="ppw-icon">f</span></a>
									<a href="#" aria-label="Twitter"><span class="ppw-icon">𝕏</span></a>
								</div>
							</div>
						<?php endif; ?>
						<?php $this->render_meta_row( 'client', esc_html__( 'Client', 'disto' ), $items[0]['client_name'] ); ?>
						<?php $this->render_meta_row( 'author', esc_html__( 'Author', 'disto' ), $items[0]['author_name'] ); ?>
					</div>
				</div>
			</div>
			<template class="ppw-meta-templates">
				<?php foreach ( $items as $idx => $item ) : ?>
					<div class="ppw-meta-template" data-index="<?php echo (int) $idx; ?>">
						<span class="ppw-title"><?php echo esc_html( $item['title'] ); ?></span>
						<span class="ppw-desc"><?php echo esc_html( wp_strip_all_tags( $item['description'] ) ); ?></span>
						<span class="ppw-tags"><?php echo esc_html( $item['custom_tags'] ); ?></span>
						<span class="ppw-category"><?php echo esc_html( $item['category'] ); ?></span>
						<span class="ppw-date"><?php echo esc_html( $item['date'] ); ?></span>
						<span class="ppw-client"><?php echo esc_html( $item['client_name'] ); ?></span>
						<span class="ppw-author"><?php echo esc_html( $item['author_name'] ); ?></span>
					</div>
				<?php endforeach; ?>
			</template>
		</div>
		<?php
	}

	private function render_skin_creative( $items, $settings ) {
		$title_html  = isset( $settings['creative_title_html'] ) ? $settings['creative_title_html'] : '';
		$subtitle    = isset( $settings['creative_subtitle'] ) ? $settings['creative_subtitle'] : '';
		$cols        = isset( $settings['creative_columns']['size'] ) ? (int) $settings['creative_columns']['size'] : 4;
		?>
		<div class="ppw-creative">
			<?php if ( $title_html || $subtitle ) : ?>
				<header class="ppw-creative-header">
					<?php if ( $title_html ) : ?>
						<h2 class="ppw-creative-title"><?php echo wp_kses_post( $title_html ); ?></h2>
					<?php endif; ?>
					<?php if ( $subtitle ) : ?>
						<p class="ppw-creative-subtitle"><?php echo wp_kses_post( $subtitle ); ?></p>
					<?php endif; ?>
				</header>
			<?php endif; ?>
			<div class="ppw-grid ppw-grid-gapless" style="--ppw-cols: <?php echo (int) $cols; ?>;">
				<?php foreach ( $items as $index => $item ) : ?>
					<div class="ppw-item ppw-creative-item" data-index="<?php echo (int) $index; ?>" data-category="<?php echo esc_attr( $item['category'] ); ?>">
						<?php $this->render_item_link_wrap( $item, $index, $settings, 'creative' ); ?>
					</div>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
	}

	private function render_skin_bento( $items, $settings ) {
		$masonry = ! empty( $settings['bento_masonry'] ) && $settings['bento_masonry'] === 'yes';
		?>
		<div class="ppw-bento <?php echo $masonry ? 'ppw-masonry' : ''; ?>">
			<div class="ppw-bento-grid" data-masonry="<?php echo $masonry ? '1' : '0'; ?>">
				<?php foreach ( $items as $index => $item ) : ?>
					<div class="ppw-item ppw-bento-item" data-index="<?php echo (int) $index; ?>" data-category="<?php echo esc_attr( $item['category'] ); ?>">
						<?php if ( ! empty( $item['category'] ) ) : ?>
							<span class="ppw-category-tag ppw-absolute-tag"><?php echo esc_html( $item['category'] ); ?></span>
						<?php endif; ?>
						<?php $this->render_item_link_wrap( $item, $index, $settings, 'bento' ); ?>
					</div>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
	}

	private function render_meta_row( $key, $label, $value ) {
		?>
		<div class="ppw-meta-row" data-meta-key="<?php echo esc_attr( $key ); ?>">
			<span class="ppw-meta-label"><?php echo esc_html( $label ); ?></span>
			<span class="ppw-meta-value"><?php echo esc_html( (string) $value ); ?></span>
		</div>
		<?php
	}

	private function render_item_image( $item, $index, $settings ) {
		$img_url = isset( $item['image']['url'] ) ? $item['image']['url'] : '';
		if ( ! $img_url ) {
			echo '<div class="ppw-img-wrap ppw-placeholder"></div>';
			return;
		}
		$alt = ! empty( $item['title'] ) ? $item['title'] : '';
		?>
		<div class="ppw-img-wrap">
			<img src="<?php echo esc_url( $img_url ); ?>" alt="<?php echo esc_attr( $alt ); ?>" loading="lazy" decoding="async" />
		</div>
		<?php
	}

	private function render_item_link_wrap( $item, $index, $settings, $skin ) {
		$img_url   = isset( $item['image']['url'] ) ? $item['image']['url'] : '';
		$ext       = isset( $item['external_link']['url'] ) ? $item['external_link']['url'] : '';
		$lightbox_url = '';
		if ( ! empty( $item['lightbox_video_url'] ) ) {
			$lightbox_url = $item['lightbox_video_url'];
		} elseif ( ! empty( $item['lightbox_image']['url'] ) ) {
			$lightbox_url = $item['lightbox_image']['url'];
		} elseif ( $img_url ) {
			$lightbox_url = $img_url;
		}

		$use_lightbox = ! empty( $lightbox_url );
		$href         = $ext ? $ext : ( $use_lightbox ? $lightbox_url : '#' );
		$link_target  = ! empty( $item['external_link']['is_external'] ) ? ' target="_blank"' : '';
		$link_rel     = ! empty( $item['external_link']['nofollow'] ) ? ' rel="nofollow"' : '';

		if ( $use_lightbox && ! $ext ) {
			$link_rel .= ' rel="noopener"';
		}
		?>
		<a href="<?php echo esc_url( $href ); ?>" class="ppw-item-inner"<?php echo $link_target . $link_rel; ?>
			<?php if ( $use_lightbox && ! $ext ) : ?>
				data-elementor-open-lightbox="yes"
				data-elementor-lightbox="slideshow"
			<?php endif; ?>
		>
			<div class="ppw-img-wrap">
				<?php if ( $img_url ) : ?>
					<img src="<?php echo esc_url( $img_url ); ?>" alt="<?php echo esc_attr( $item['title'] ); ?>" loading="lazy" decoding="async" />
				<?php else : ?>
					<div class="ppw-placeholder"></div>
				<?php endif; ?>
			</div>
			<div class="ppw-overlay">
				<h3 class="ppw-title"><?php echo esc_html( $item['title'] ); ?></h3>
				<?php if ( ! empty( $item['category'] ) ) : ?>
					<span class="ppw-category-tag"><?php echo esc_html( $item['category'] ); ?></span>
				<?php endif; ?>
				<?php if ( ! empty( $item['button_text'] ) ) : ?>
					<span class="ppw-btn"><?php echo esc_html( $item['button_text'] ); ?></span>
				<?php endif; ?>
			</div>
		</a>
		<?php
	}

}
