<?php
/**
 * News Sidebar Widget (Elementor Widget)
 *
 * ویجت سایدبار خبری - فقط بخش سایدبار با لیست عمودی پست‌ها
 *
 * کاملاً RTL و سازگار با زبان فارسی
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;

class Disto_News_Sidebar_Widget extends Widget_Base {

	public function get_name() {
		return 'disto_news_sidebar';
	}

	public function get_title() {
		return esc_html__( 'سایدبار خبری', 'disto' );
	}

	public function get_icon() {
		return 'eicon-sidebar';
	}

	public function get_categories() {
		return array( 'disto-elements' );
	}

	public function get_style_depends() {
		return array( 'disto-child-news-sidebar' );
	}

	public function get_script_depends() {
		return array();
	}

	protected function register_controls() {
		// ============================================
		// Content Tab - Sidebar Section
		// ============================================
		$this->start_controls_section(
			'content_sidebar_section',
			array(
				'label' => esc_html__( 'تنظیمات محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'sidebar_title',
			array(
				'label'   => esc_html__( 'عنوان سایدبار', 'disto' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Blog And News', 'disto' ),
			)
		);

		// Categories options
		$terms       = get_terms(
			array(
				'taxonomy'   => 'category',
				'hide_empty' => false,
			)
		);
		$cat_options = array();
		if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
			foreach ( $terms as $term ) {
				$cat_options[ (int) $term->term_id ] = $term->name;
			}
		}

		$this->add_control(
			'sidebar_categories',
			array(
				'label'       => esc_html__( 'دسته‌بندی پست‌ها', 'disto' ),
				'type'        => Controls_Manager::SELECT2,
				'options'     => $cat_options,
				'multiple'    => true,
				'label_block' => true,
				'description' => esc_html__( 'می‌توانید یک یا چند دسته انتخاب کنید. در صورت عدم انتخاب، همه دسته‌ها نمایش داده می‌شوند.', 'disto' ),
			)
		);

		$this->add_control(
			'sidebar_posts_count',
			array(
				'label'   => esc_html__( 'تعداد پست‌ها', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 20,
				'step'    => 1,
				'default' => 4,
			)
		);

		$this->add_control(
			'sidebar_order_by',
			array(
				'label'   => esc_html__( 'مرتب‌سازی', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'latest',
				'options' => array(
					'latest'   => esc_html__( 'جدیدترین', 'disto' ),
					'oldest'   => esc_html__( 'قدیمی‌ترین', 'disto' ),
					'alphabet' => esc_html__( 'بر اساس حروف الفبا (A-Z)', 'disto' ),
					'views'    => esc_html__( 'بیشترین بازدید', 'disto' ),
					'random'   => esc_html__( 'تصادفی', 'disto' ),
				),
			)
		);

		$this->add_control(
			'show_image',
			array(
				'label'   => esc_html__( 'نمایش تصویر', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_date',
			array(
				'label'   => esc_html__( 'نمایش تاریخ', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Widget Container
		// ============================================
		$this->start_controls_section(
			'style_container_section',
			array(
				'label' => esc_html__( 'کانتینر ویجت', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'container_background',
				'label'    => esc_html__( 'پس‌زمینه', 'disto' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .news-sidebar-container',
			)
		);

		$this->add_responsive_control(
			'container_padding',
			array(
				'label'      => esc_html__( 'Padding', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'container_margin',
			array(
				'label'      => esc_html__( 'Margin', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'container_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-container' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'container_box_shadow',
				'label'    => esc_html__( 'Box Shadow', 'disto' ),
				'selector' => '{{WRAPPER}} .news-sidebar-container',
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Sidebar Section
		// ============================================
		$this->start_controls_section(
			'style_sidebar_section',
			array(
				'label' => esc_html__( 'سایدبار', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		// Sidebar Title
		$this->add_control(
			'sidebar_title_heading',
			array(
				'label'     => esc_html__( 'عنوان سایدبار', 'disto' ),
				'type'      => Controls_Manager::HEADING,
			)
		);

		$this->add_control(
			'sidebar_title_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .news-sidebar-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'sidebar_title_alignment',
			array(
				'label'   => esc_html__( 'تراز', 'disto' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => array(
					'right'  => array(
						'title' => esc_html__( 'راست', 'disto' ),
						'icon'  => 'eicon-text-align-right',
					),
					'center' => array(
						'title' => esc_html__( 'وسط', 'disto' ),
						'icon'  => 'eicon-text-align-center',
					),
					'left'   => array(
						'title' => esc_html__( 'چپ', 'disto' ),
						'icon'  => 'eicon-text-align-left',
					),
				),
				'default' => 'right',
				'toggle'  => true,
				'selectors' => array(
					'{{WRAPPER}} .news-sidebar-title' => 'text-align: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sidebar_title_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .news-sidebar-title',
			)
		);

		$this->add_responsive_control(
			'sidebar_title_margin',
			array(
				'label'      => esc_html__( 'فاصله پایین', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// Sidebar List Items
		$this->add_control(
			'sidebar_item_heading',
			array(
				'label'     => esc_html__( 'آیتم‌های لیست', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'sidebar_item_bg',
			array(
				'label'     => esc_html__( 'پس‌زمینه آیتم', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .news-sidebar-item' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'sidebar_item_hover_bg',
			array(
				'label'     => esc_html__( 'پس‌زمینه آیتم (Hover)', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .news-sidebar-item:hover' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_item_padding',
			array(
				'label'      => esc_html__( 'Padding آیتم', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_item_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius آیتم', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_item_spacing',
			array(
				'label'      => esc_html__( 'فاصله بین آیتم‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-item' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'sidebar_item_box_shadow',
				'label'    => esc_html__( 'Box Shadow آیتم', 'disto' ),
				'selector' => '{{WRAPPER}} .news-sidebar-item',
			)
		);

		// Sidebar Image
		$this->add_control(
			'sidebar_image_heading',
			array(
				'label'     => esc_html__( 'تصویر', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_responsive_control(
			'sidebar_image_width',
			array(
				'label'      => esc_html__( 'عرض', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', '%' ),
				'range'      => array(
					'px' => array(
						'min' => 50,
						'max' => 300,
					),
					'%'  => array(
						'min' => 10,
						'max' => 100,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-item .news-sidebar-image' => 'width: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_image_height',
			array(
				'label'      => esc_html__( 'ارتفاع', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 50,
						'max' => 300,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-item .news-sidebar-image' => 'height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_image_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-item .news-sidebar-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_image_spacing',
			array(
				'label'      => esc_html__( 'فاصله از متن', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .news-sidebar-item .news-sidebar-image' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// Sidebar Date
		$this->add_control(
			'sidebar_date_heading',
			array(
				'label'     => esc_html__( 'تاریخ', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'sidebar_date_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .news-sidebar-item .news-sidebar-date' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sidebar_date_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .news-sidebar-item .news-sidebar-date',
			)
		);

		// Sidebar Title
		$this->add_control(
			'sidebar_item_title_heading',
			array(
				'label'     => esc_html__( 'عنوان آیتم', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'sidebar_item_title_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .news-sidebar-item .news-sidebar-item-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'sidebar_item_title_hover_color',
			array(
				'label'     => esc_html__( 'رنگ Hover', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .news-sidebar-item:hover .news-sidebar-item-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sidebar_item_title_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .news-sidebar-item .news-sidebar-item-title',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Get posts query arguments
	 *
	 * @param array $settings Widget settings.
	 * @return array Query arguments.
	 */
	private function get_posts_query_args( $settings ) {
		$args = array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'ignore_sticky_posts' => true,
			'no_found_rows'       => true,
		);

		// Posts count
		$posts_count = absint( $settings['sidebar_posts_count'] );
		$args['posts_per_page'] = $posts_count > 0 ? $posts_count : 4;

		// Categories
		if ( ! empty( $settings['sidebar_categories'] ) && is_array( $settings['sidebar_categories'] ) ) {
			$args['category__in'] = array_map( 'absint', $settings['sidebar_categories'] );
		}

		// Order by
		$order_by = $settings['sidebar_order_by'];

		switch ( $order_by ) {
			case 'oldest':
				$args['orderby'] = 'date';
				$args['order']   = 'ASC';
				break;
			case 'alphabet':
				$args['orderby'] = 'title';
				$args['order']   = 'ASC';
				break;
			case 'views':
				$args['orderby']  = 'meta_value_num';
				$args['meta_key'] = 'post_views_count';
				$args['order']    = 'DESC';
				break;
			case 'random':
				$args['orderby'] = 'rand';
				break;
			case 'latest':
			default:
				$args['orderby'] = 'date';
				$args['order']   = 'DESC';
				break;
		}

		return $args;
	}

	/**
	 * Render widget output
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		// Sidebar posts query
		$sidebar_query = new \WP_Query( $this->get_posts_query_args( $settings ) );

		$widget_id = 'news-sidebar-' . $this->get_id();
		?>
		<div id="<?php echo esc_attr( $widget_id ); ?>" class="news-sidebar-container" dir="rtl">
			<?php if ( ! empty( $settings['sidebar_title'] ) ) : ?>
				<h2 class="news-sidebar-title"><?php echo esc_html( $settings['sidebar_title'] ); ?></h2>
			<?php endif; ?>
			<?php if ( $sidebar_query->have_posts() ) : ?>
				<div class="news-sidebar-list">
					<?php
					while ( $sidebar_query->have_posts() ) :
						$sidebar_query->the_post();
						$post_id = get_the_ID();
						$title   = get_the_title();
						$link    = get_permalink();
						?>
						<a href="<?php echo esc_url( $link ); ?>" class="news-sidebar-item">
							<?php if ( ! empty( $settings['show_image'] ) && 'yes' === $settings['show_image'] && has_post_thumbnail( $post_id ) ) : ?>
								<div class="news-sidebar-image">
									<?php echo get_the_post_thumbnail( $post_id, 'thumbnail', array( 'class' => 'news-sidebar-img' ) ); ?>
								</div>
							<?php endif; ?>
							<div class="news-sidebar-content">
								<?php if ( ! empty( $settings['show_date'] ) && 'yes' === $settings['show_date'] ) : ?>
									<span class="news-sidebar-date"><?php echo esc_html( get_the_date( 'F j, Y', $post_id ) ); ?></span>
								<?php endif; ?>
								<h4 class="news-sidebar-item-title"><?php echo esc_html( $title ); ?></h4>
							</div>
						</a>
						<?php
					endwhile;
					wp_reset_postdata();
					?>
				</div>
			<?php else : ?>
				<p class="news-sidebar-no-posts"><?php esc_html_e( 'پستی یافت نشد.', 'disto' ); ?></p>
			<?php endif; ?>
		</div>
		<?php
	}
}



