<?php
/**
 * FAQ (سوالات متداول) – ویجت Elementor راست‌چین با تصویر و آکاردئون
 *
 * چیدمان RTL: دسکتاپ = تصویر راست، سوالات چپ؛ موبایل = تصویر بالا، سوالات پایین.
 * هر سوال تصویر و لینک «مشاهده بیشتر» اختصاصی دارد.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Repeater;

/**
 * پیکربندی پیش‌فرض ویجت – قابل تغییر از پنل Elementor یا از اینجا.
 */
class Disto_FAQ_Widget_Config {
	public static function get_defaults() {
		return array(
			'primary_color'     => '#FFD700',
			'tag_text'          => 'FAQS',
			'title'             => 'سوالات رخت‌شویی شما، پاسخ داده شد!',
			'description'       => 'لورم ایپسوم متن ساختگی با تولید سادگی نامفهوم از صنعت چاپ و با استفاده از طراحان گرافیک است.',
			'read_more_text'    => 'مشاهده بیشتر',
			'font_family'       => 'Vazir, Tahoma, sans-serif',
			'title_font_size'   => array( 'unit' => 'px', 'size' => 28 ),
			'question_font_size' => array( 'unit' => 'px', 'size' => 16 ),
			'image_border_radius' => '3rem',
		);
	}
}

class Disto_FAQ_Widget extends Widget_Base {

	public function get_name() {
		return 'disto_faq';
	}

	public function get_title() {
		return esc_html__( 'سوالات متداول (FAQ)', 'disto' );
	}

	public function get_icon() {
		return 'eicon-help';
	}

	public function get_categories() {
		return array( 'disto-elements' );
	}

	public function get_style_depends() {
		return array( 'disto-child-faq-widget' );
	}

	public function get_script_depends() {
		return array( 'disto-child-faq-widget' );
	}

	protected function register_controls() {
		$config = Disto_FAQ_Widget_Config::get_defaults();

		// ─── محتوا: تگ، عنوان، توضیح ─────────────────────────
		$this->start_controls_section(
			'section_header',
			array(
				'label' => esc_html__( 'هدر و توضیح', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'tag_text',
			array(
				'label'       => esc_html__( 'متن تگ (مثلاً FAQS)', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => $config['tag_text'],
			)
		);

		$this->add_control(
			'title',
			array(
				'label'       => esc_html__( 'عنوان اصلی', 'disto' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => $config['title'],
				'rows'        => 2,
			)
		);

		$this->add_control(
			'description',
			array(
				'label'       => esc_html__( 'توضیح کوتاه', 'disto' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => $config['description'],
				'rows'        => 3,
			)
		);

		$this->add_control(
			'read_more_text',
			array(
				'label'       => esc_html__( 'متن لینک «مشاهده بیشتر»', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => $config['read_more_text'],
			)
		);

		$this->end_controls_section();

		// ─── سوالات (Repeater) ───────────────────────────────
		$repeater = new Repeater();

		$repeater->add_control(
			'question',
			array(
				'label'       => esc_html__( 'سوال', 'disto' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => esc_html__( 'سرویس دریافت و تحویل چگونه است؟', 'disto' ),
			)
		);

		$repeater->add_control(
			'answer',
			array(
				'label'       => esc_html__( 'پاسخ', 'disto' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => esc_html__( 'متن پاسخ این سوال را اینجا وارد کنید.', 'disto' ),
			)
		);

		$repeater->add_control(
			'image',
			array(
				'label'   => esc_html__( 'تصویر این سوال', 'disto' ),
				'type'    => Controls_Manager::MEDIA,
				'default' => array(),
			)
		);

		$repeater->add_control(
			'link',
			array(
				'label'         => esc_html__( 'لینک «مشاهده بیشتر»', 'disto' ),
				'type'          => Controls_Manager::URL,
				'placeholder'   => 'https://',
				'show_external' => true,
				'default'       => array(
					'url'         => '',
					'is_external' => false,
					'nofollow'    => false,
				),
			)
		);

		$this->start_controls_section(
			'section_items',
			array(
				'label' => esc_html__( 'سوالات', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'faq_items',
			array(
				'label'       => esc_html__( 'لیست سوالات', 'disto' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => array(
					array(
						'question' => esc_html__( 'سرویس دریافت و تحویل چگونه است؟', 'disto' ),
						'answer'   => esc_html__( 'پاسخ نمونه برای سوال اول.', 'disto' ),
					),
					array(
						'question' => esc_html__( 'زمان تحویل استاندارد چند روز است؟', 'disto' ),
						'answer'   => esc_html__( 'پاسخ نمونه برای سوال دوم.', 'disto' ),
					),
					array(
						'question' => esc_html__( 'آیا خدمات رخت‌شویی همان روز دارید؟', 'disto' ),
						'answer'   => esc_html__( 'پاسخ نمونه برای سوال سوم.', 'disto' ),
					),
				),
				'title_field' => '{{{ question }}}',
			)
		);

		$this->end_controls_section();

		// ─── استایل: رنگ و تایپوگرافی ───────────────────────
		$this->start_controls_section(
			'section_style',
			array(
				'label' => esc_html__( 'ظاهر', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'primary_color',
			array(
				'label'     => esc_html__( 'رنگ اصلی (تگ و سوال فعال)', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => $config['primary_color'],
				'selectors' => array(
					'{{WRAPPER}} .disto-faq-tag'                    => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .disto-faq-item.disto-faq-open .disto-faq-question-wrap' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_title',
				'label'    => esc_html__( 'عنوان اصلی', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-faq-title',
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'رنگ عنوان', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#111',
				'selectors' => array(
					'{{WRAPPER}} .disto-faq-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_description',
				'label'    => esc_html__( 'توضیح', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-faq-description',
			)
		);

		$this->add_control(
			'description_color',
			array(
				'label'     => esc_html__( 'رنگ توضیح', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#666',
				'selectors' => array(
					'{{WRAPPER}} .disto-faq-description' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_question',
				'label'    => esc_html__( 'متن سوال', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-faq-question-text',
			)
		);

		$this->add_control(
			'question_text_color',
			array(
				'label'     => esc_html__( 'رنگ متن سوال', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#111',
				'selectors' => array(
					'{{WRAPPER}} .disto-faq-question-text' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'typography_answer',
				'label'    => esc_html__( 'متن پاسخ', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-faq-answer',
			)
		);

		$this->add_control(
			'answer_color',
			array(
				'label'     => esc_html__( 'رنگ پاسخ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#444',
				'selectors' => array(
					'{{WRAPPER}} .disto-faq-answer' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'image_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه تصویر', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'rem' ),
				'range'      => array(
					'px'  => array( 'min' => 0, 'max' => 120 ),
					'rem' => array( 'min' => 0, 'max' => 8 ),
				),
				'default'    => array( 'unit' => 'rem', 'size' => 3 ),
				'selectors'  => array(
					'{{WRAPPER}} .disto-faq-image-wrap' => 'border-radius: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$s = $this->get_settings_for_display();
		$items = ! empty( $s['faq_items'] ) ? $s['faq_items'] : array();

		if ( empty( $items ) ) {
			echo '<p class="disto-faq-empty">' . esc_html__( 'لطفاً حداقل یک سوال در تب «سوالات» اضافه کنید.', 'disto' ) . '</p>';
			return;
		}

		$widget_id = 'disto-faq-' . (string) $this->get_id();
		?>
		<div class="disto-faq-widget" id="<?php echo esc_attr( $widget_id ); ?>" dir="rtl">
			<div class="disto-faq-inner">
				<!-- ستون چپ: سوالات (در RTL سمت چپ صفحه) -->
				<div class="disto-faq-content">
					<?php if ( ! empty( $s['tag_text'] ) ) : ?>
						<span class="disto-faq-tag"><?php echo esc_html( $s['tag_text'] ); ?></span>
					<?php endif; ?>
					<?php if ( ! empty( $s['title'] ) ) : ?>
						<h2 class="disto-faq-title"><?php echo esc_html( $s['title'] ); ?></h2>
					<?php endif; ?>
					<?php if ( ! empty( $s['description'] ) ) : ?>
						<p class="disto-faq-description"><?php echo esc_html( $s['description'] ); ?></p>
					<?php endif; ?>

					<div class="disto-faq-list">
						<?php
						foreach ( $items as $i => $item ) {
							$open_class = ( 0 === $i ) ? ' disto-faq-open' : '';
							$q         = isset( $item['question'] ) ? $item['question'] : '';
							$a         = isset( $item['answer'] ) ? $item['answer'] : '';
							$img       = isset( $item['image'] ) ? $item['image'] : array();
							$thumb_id  = ! empty( $img['id'] ) ? (int) $img['id'] : 0;
							$img_url   = '';
							if ( $thumb_id ) {
								$img_data = wp_get_attachment_image_src( $thumb_id, 'large' );
								if ( ! empty( $img_data[0] ) ) {
									$img_url = $img_data[0];
								}
							}
							if ( empty( $img_url ) && ! empty( $img['url'] ) ) {
								$img_url = $img['url'];
							}
							$link_arr = isset( $item['link'] ) ? $item['link'] : array();
							$link_url = ! empty( $link_arr['url'] ) ? $link_arr['url'] : '';
							$link_attrs = '';
							if ( ! empty( $link_arr['is_external'] ) ) {
								$link_attrs .= ' target="_blank"';
							}
							$rel_parts = array();
							if ( ! empty( $link_arr['is_external'] ) ) {
								$rel_parts[] = 'noopener';
								$rel_parts[] = 'noreferrer';
							}
							if ( ! empty( $link_arr['nofollow'] ) ) {
								$rel_parts[] = 'nofollow';
							}
							if ( ! empty( $rel_parts ) ) {
								$link_attrs .= ' rel="' . esc_attr( implode( ' ', $rel_parts ) ) . '"';
							}
							?>
							<div class="disto-faq-item<?php echo esc_attr( $open_class ); ?>" data-index="<?php echo (int) $i; ?>" data-image="<?php echo esc_url( $img_url ); ?>">
								<div class="disto-faq-question-wrap">
									<span class="disto-faq-question-text"><?php echo esc_html( $q ); ?></span>
									<span class="disto-faq-icon" aria-hidden="true">+</span>
								</div>
								<div class="disto-faq-answer-wrap">
									<div class="disto-faq-answer"><?php echo wp_kses_post( nl2br( $a ) ); ?></div>
									<?php if ( $link_url && ! empty( $s['read_more_text'] ) ) : ?>
										<a href="<?php echo esc_url( $link_url ); ?>" class="disto-faq-read-more"<?php echo $link_attrs; ?>><?php echo esc_html( $s['read_more_text'] ); ?></a>
									<?php endif; ?>
								</div>
							</div>
						<?php } ?>
					</div>
				</div>

				<!-- ستون راست: تصویر (در RTL سمت راست صفحه) -->
				<div class="disto-faq-image-col">
					<div class="disto-faq-image-wrap">
						<div class="disto-faq-image-inner">
							<?php
							foreach ( $items as $i => $item ) {
								$img   = isset( $item['image'] ) ? $item['image'] : array();
								$tid   = ! empty( $img['id'] ) ? (int) $img['id'] : 0;
								$url   = '';
								if ( $tid ) {
									$img_data = wp_get_attachment_image_src( $tid, 'large' );
									if ( ! empty( $img_data[0] ) ) {
										$url = $img_data[0];
									}
								}
								if ( empty( $url ) && ! empty( $img['url'] ) ) {
									$url = $img['url'];
								}
								if ( empty( $url ) ) {
									continue;
								}
								$active_class = ( 0 === $i ) ? ' disto-faq-img-active' : '';
								?>
								<img src="<?php echo esc_url( $url ); ?>" alt="" class="disto-faq-img<?php echo esc_attr( $active_class ); ?>" data-index="<?php echo (int) $i; ?>" />
							<?php } ?>
							<?php
							// اگر هیچ تصویری نبود، یک placeholder خالی تا لایه نگه داشته شود
							$has_any = false;
							foreach ( $items as $item ) {
								$img = isset( $item['image'] ) ? $item['image'] : array();
								if ( ! empty( $img['id'] ) || ! empty( $img['url'] ) ) {
									$has_any = true;
									break;
								}
							}
							if ( ! $has_any ) :
								?>
								<span class="disto-faq-img-placeholder"></span>
							<?php endif; ?>
						</div>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}
