<?php
/**
 * Simple sample Elementor widget for Disto category.
 *
 * You can duplicate this class and change:
 * - class name
 * - get_name()
 * - get_title()
 * - get_icon()
 * - controls & render()
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Make sure Elementor is loaded.
if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Group_Control_Typography;

class Disto_Sample_Widget extends Widget_Base {

	/**
	 * Widget slug (used in shortcode / internal ID).
	 *
	 * @return string
	 */
	public function get_name() {
		return 'disto_news_card';
	}

	/**
	 * Widget title (shown in Elementor panel).
	 *
	 * @return string
	 */
	public function get_title() {
		return esc_html__( 'کارت خبر دیستو', 'disto' );
	}

	/**
	 * Icon in Elementor panel (Font Awesome).
	 *
	 * @return string
	 */
	public function get_icon() {
		return 'eicon-star';
	}

	/**
	 * Categories this widget belongs to.
	 * It must match the category slug we register: disto-elements.
	 *
	 * @return array
	 */
	public function get_categories() {
		return array( 'disto-elements' );
	}

	/**
	 * Register widget controls (settings in Elementor panel).
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'mode',
			array(
				'label'   => esc_html__( 'منبع محتوا', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'latest',
				'options' => array(
					'latest'  => esc_html__( 'آخرین نوشته‌ها', 'disto' ),
					'category'=> esc_html__( 'بر اساس دسته‌بندی', 'disto' ),
					'manual'  => esc_html__( 'انتخاب دستی نوشته‌ها', 'disto' ),
				),
			)
		);

		$this->add_control(
			'layout',
			array(
				'label'   => esc_html__( 'تم کارت', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'layout_1',
				'options' => array(
					'layout_1' => esc_html__( 'تم ۱ (کارت با منوی کشویی)', 'disto' ),
					'layout_2' => esc_html__( 'تم ۲ (کارت با دکمه "بیشتر بخوانید")', 'disto' ),
				),
			)
		);

		$this->add_control(
			'columns',
			array(
				'label'       => esc_html__( 'تعداد کارت در هر ردیف', 'disto' ),
				'type'        => Controls_Manager::NUMBER,
				'min'         => 1,
				'max'         => 6,
				'step'        => 1,
				'default'     => 3,
				'description' => esc_html__( 'حداکثر ۶ کارت در هر ردیف.', 'disto' ),
			)
		);

		$this->add_control(
			'rows',
			array(
				'label'       => esc_html__( 'تعداد ردیف', 'disto' ),
				'type'        => Controls_Manager::NUMBER,
				'min'         => 1,
				'max'         => 4,
				'step'        => 1,
				'default'     => 1,
				'description' => esc_html__( 'در حالت آخرین نوشته‌ها / دسته‌بندی، تعداد کل کارت‌ها = ردیف × کارت در هر ردیف.', 'disto' ),
				'condition'   => array(
					'mode!' => 'manual',
				),
			)
		);

		$this->add_control(
			'read_more_text',
			array(
				'label'       => esc_html__( 'متن «ادامه مطلب»', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'ادامه مطلب', 'disto' ),
				'placeholder' => esc_html__( 'مثلاً: ادامه مطلب…', 'disto' ),
			)
		);

		// Category selector (for "category" mode).
		$categories   = get_categories(
			array(
				'hide_empty' => false,
			)
		);
		$cat_options  = array( '' => esc_html__( 'همه دسته‌ها', 'disto' ) );
		if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
			foreach ( $categories as $cat ) {
				$cat_options[ $cat->term_id ] = $cat->name;
			}
		}

		$this->add_control(
			'category_id',
			array(
				'label'     => esc_html__( 'دسته نوشته‌ها', 'disto' ),
				'type'      => Controls_Manager::SELECT2,
				'options'   => $cat_options,
				'default'   => '',
				'condition' => array(
					'mode' => 'category',
				),
			)
		);

		// Manual selection via repeater (each item is one card/post).
		$repeater = new Repeater();

		// Build simple list of recent posts for dropdown (for convenience).
		$posts_dropdown = array();
		$recent_posts   = get_posts(
			array(
				'post_type'      => 'post',
				'posts_per_page' => 50,
				'post_status'    => 'publish',
			)
		);
		if ( ! empty( $recent_posts ) ) {
			foreach ( $recent_posts as $post_obj ) {
				/* translators: %1$s: post ID, %2$s: post title */
				$posts_dropdown[ $post_obj->ID ] = sprintf( '#%1$d - %2$s', $post_obj->ID, $post_obj->post_title );
			}
		}

		$repeater->add_control(
			'post_id',
			array(
				'label'   => esc_html__( 'انتخاب نوشته', 'disto' ),
				'type'    => Controls_Manager::SELECT2,
				'options' => $posts_dropdown,
			)
		);

		$repeater->add_control(
			'item_layout',
			array(
				'label'   => esc_html__( 'تم این کارت', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'default',
				'options' => array(
					'default'  => esc_html__( 'استفاده از تم اصلی ویجت', 'disto' ),
					'layout_1' => esc_html__( 'تم ۱', 'disto' ),
					'layout_2' => esc_html__( 'تم ۲', 'disto' ),
				),
			)
		);

		$this->add_control(
			'manual_items',
			array(
				'label'       => esc_html__( 'کارت‌ها (انتخاب دستی)', 'disto' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_field' => '{{{ post_id }}}',
				'condition'   => array(
					'mode' => 'manual',
				),
			)
		);

		// Simple fallback title (not required but kept for backward-compat).
		$this->add_control(
			'title',
			array(
				'label'       => esc_html__( 'عنوان', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'این یک ویجت نمونه از دیستو است', 'disto' ),
				'placeholder' => esc_html__( 'متن خود را اینجا بنویسید', 'disto' ),
			)
		);

		$this->end_controls_section();

		/**
		 * =========================
		 * Style controls
		 * =========================
		 */
		$this->start_controls_section(
			'style_typography',
			array(
				'label' => esc_html__( 'استایل (تایپوگرافی و رنگ)', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		// Author.
		$this->add_control(
			'author_color',
			array(
				'label'     => esc_html__( 'رنگ نویسنده', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .disto-news-cards .author' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'author_typography',
				'label'    => esc_html__( 'تایپوگرافی نویسنده', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-news-cards .author',
			)
		);

		// Title.
		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'رنگ عنوان', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .disto-news-cards .title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'label'    => esc_html__( 'تایپوگرافی عنوان', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-news-cards .title',
			)
		);

		// Excerpt.
		$this->add_control(
			'excerpt_color',
			array(
				'label'     => esc_html__( 'رنگ خلاصه مطلب', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .disto-news-cards .text' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'excerpt_typography',
				'label'    => esc_html__( 'تایپوگرافی خلاصه مطلب', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-news-cards .text',
			)
		);

		// Date (day/month/year).
		$this->add_control(
			'date_color',
			array(
				'label'     => esc_html__( 'رنگ تاریخ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .disto-news-cards .date, {{WRAPPER}} .disto-news-cards .date span' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'date_typography',
				'label'    => esc_html__( 'تایپوگرافی تاریخ', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-news-cards .date',
			)
		);

		// Read more.
		$this->add_control(
			'read_more_color',
			array(
				'label'     => esc_html__( 'رنگ «ادامه مطلب»', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .disto-news-cards .disto-read-more' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'read_more_typography',
				'label'    => esc_html__( 'تایپوگرافی «ادامه مطلب»', 'disto' ),
				'selector' => '{{WRAPPER}} .disto-news-cards .disto-read-more',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$mode     = isset( $settings['mode'] ) ? $settings['mode'] : 'latest';
		$layout   = isset( $settings['layout'] ) ? $settings['layout'] : 'layout_1';

		// Columns / rows settings.
		$columns = ! empty( $settings['columns'] ) ? absint( $settings['columns'] ) : 3;
		if ( $columns < 1 ) {
			$columns = 1;
		}
		if ( $columns > 6 ) {
			$columns = 6;
		}

		$rows = ! empty( $settings['rows'] ) ? absint( $settings['rows'] ) : 1;
		if ( $rows < 1 ) {
			$rows = 1;
		}
		if ( $rows > 4 ) {
			$rows = 4;
		}

		$cards = array();

		if ( 'manual' === $mode && ! empty( $settings['manual_items'] ) && is_array( $settings['manual_items'] ) ) {
			// Manual list of posts.
			foreach ( $settings['manual_items'] as $item ) {
				if ( empty( $item['post_id'] ) ) {
					continue;
				}
				$cards[] = array(
					'post_id'    => (int) $item['post_id'],
					'item_layout'=> isset( $item['item_layout'] ) ? $item['item_layout'] : 'default',
				);
			}
		} else {
			// Automatic query (latest or by category).
			$total_cards = $columns * $rows;
			if ( $total_cards < 1 ) {
				$total_cards = 1;
			}
			if ( $total_cards > 24 ) {
				$total_cards = 24;
			}

			$args = array(
				'post_type'           => 'post',
				'posts_per_page'      => $total_cards,
				'post_status'         => 'publish',
				'ignore_sticky_posts' => true,
			);

			if ( 'category' === $mode && ! empty( $settings['category_id'] ) ) {
				$args['cat'] = (int) $settings['category_id'];
			}

			$query = new \WP_Query( $args );

			if ( $query->have_posts() ) {
				while ( $query->have_posts() ) {
					$query->the_post();
					$cards[] = array(
						'post_id'     => get_the_ID(),
						'item_layout' => 'default',
					);
				}
				wp_reset_postdata();
			}
		}

		if ( empty( $cards ) ) {
			return;
		}

		$wrapper_classes = array(
			'disto-news-cards',
			'content-wrapper',
			'disto-cols-' . $columns,
		);

		echo '<div class="' . esc_attr( implode( ' ', $wrapper_classes ) ) . '">';

		$index = 0;
		foreach ( $cards as $card ) {
			$index++;
			$post_id     = $card['post_id'];
			$item_layout = $card['item_layout'];

			if ( 'default' === $item_layout || empty( $item_layout ) ) {
				$item_layout = $layout;
			}

			$this->render_single_card( $post_id, $item_layout, $index, $settings );
		}

		echo '</div>';
	}

	/**
	 * Render a single card (layout 1 or 2) using the HTML structure you provided.
	 *
	 * @param int    $post_id Post ID.
	 * @param string $layout  Layout key: layout_1 or layout_2.
	 * @param int    $index   1-based index (used for unique IDs).
	 */
	protected function render_single_card( $post_id, $layout, $index, $settings ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return;
		}

		setup_postdata( $post );

		$permalink = get_permalink( $post );
		$title     = get_the_title( $post );
		$author    = get_the_author_meta( 'display_name', $post->post_author );
		$post_date = get_the_date( 'j F Y', $post );
		$excerpt   = get_the_excerpt( $post );
		$comments  = get_comments_number( $post );

		$thumb_url = get_the_post_thumbnail_url( $post, 'large' );

		// Normalize excerpt: always a short, 2‑line style summary without broken HTML.
		$excerpt = is_string( $excerpt ) ? wp_strip_all_tags( $excerpt ) : '';
		$excerpt = preg_replace( '/\s+/', ' ', $excerpt );
		$excerpt = trim( $excerpt );
		if ( '' === $excerpt ) {
			$excerpt = wp_strip_all_tags( $post->post_content );
		}
		$excerpt = wp_trim_words( $excerpt, 24, '…' );

		$read_more_text = isset( $settings['read_more_text'] ) && is_string( $settings['read_more_text'] ) && $settings['read_more_text'] !== ''
			? $settings['read_more_text']
			: esc_html__( 'ادامه مطلب', 'disto' );

		?>
		<div class="news-card">
			<a class="news-card__card-link" href="<?php echo esc_url( $permalink ); ?>"></a>
			<?php if ( $thumb_url ) : ?>
				<img class="news-card__image" src="<?php echo esc_url( $thumb_url ); ?>" alt="<?php echo esc_attr( $title ); ?>" loading="lazy" decoding="async" />
			<?php else : ?>
				<div class="news-card__image news-card__image--placeholder"></div>
			<?php endif; ?>

			<div class="news-card__text-wrapper">
				<h2 class="news-card__title"><?php echo esc_html( $title ); ?></h2>

				<div class="news-card__post-date">
					<?php echo esc_html( $post_date ); ?>
				</div>

				<div class="news-card__details-wrapper">
					<p class="news-card__excerpt"><?php echo esc_html( $excerpt ); ?></p>
					<span class="news-card__read-more">
						<?php echo esc_html( $read_more_text ); ?>
					</span>
				</div>
			</div>
		</div>
		<?php

		wp_reset_postdata();
	}
}


