<?php
/**
 * Breaking News Ticker Pro RTL (Elementor Widget)
 *
 * Lightweight RTL-aware breaking news bar for Persian sites.
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;

class Disto_Breaking_News_Ticker_Pro_Widget extends Widget_Base {

	public function get_name() {
		return 'rbn_breaking_news_ticker_pro';
	}

	public function get_title() {
		return esc_html__( 'Breaking News Ticker Pro RTL', 'disto' );
	}

	public function get_icon() {
		return 'eicon-mega-menu';
	}

	public function get_categories() {
		return array( 'disto-elements' );
	}

	public function get_style_depends() {
		return array( 'disto-child-rbn' );
	}

	public function get_script_depends() {
		return array( 'disto-child-rbn' );
	}

	protected function register_controls() {
		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'تنظیمات محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		// Categories.
		$terms       = get_terms(
			array(
				'taxonomy'   => 'category',
				'hide_empty' => false,
			)
		);
		$cat_options = array(
			0 => esc_html__( 'همه دسته‌ها', 'disto' ),
		);
		if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
			foreach ( $terms as $term ) {
				$cat_options[ (int) $term->term_id ] = $term->name;
			}
		}

		$this->add_control(
			'category_id',
			array(
				'label'   => esc_html__( 'دسته نوشته‌ها', 'disto' ),
				'type'    => Controls_Manager::SELECT2,
				'options' => $cat_options,
				'default' => 0,
			)
		);

		$this->add_control(
			'posts_count',
			array(
				'label'   => esc_html__( 'تعداد نوشته‌ها', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 20,
				'step'    => 1,
				'default' => 6,
			)
		);

		$this->add_control(
			'order_mode',
			array(
				'label'   => esc_html__( 'مرتب‌سازی', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'latest',
				'options' => array(
					'latest' => esc_html__( 'جدیدترین', 'disto' ),
					'random' => esc_html__( 'تصادفی', 'disto' ),
				),
			)
		);

		$this->add_control(
			'show_date',
			array(
				'label'   => esc_html__( 'نمایش تاریخ', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_category',
			array(
				'label'   => esc_html__( 'نمایش دسته', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'no',
			)
		);

		$this->add_control(
			'enable_autoplay',
			array(
				'label'   => esc_html__( 'اتوماتیک (Autoplay)', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'autoplay_speed',
			array(
				'label'     => esc_html__( 'سرعت تعویض (میلی‌ثانیه)', 'disto' ),
				'type'      => Controls_Manager::NUMBER,
				'min'       => 1000,
				'max'       => 15000,
				'step'      => 250,
				'default'   => 3500,
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->add_control(
			'scroll_mode',
			array(
				'label'   => esc_html__( 'حالت حرکت', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'slide',
				'options' => array(
					'slide'   => esc_html__( 'اسلاید (هر بار یک تیتر)', 'disto' ),
					'marquee' => esc_html__( 'مارکی (حرکت پیوسته)', 'disto' ),
				),
			)
		);

		$this->add_control(
			'pause_on_hover',
			array(
				'label'   => esc_html__( 'توقف با هاور', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'open_new_tab',
			array(
				'label'   => esc_html__( 'باز شدن لینک در تب جدید', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'no',
			)
		);

		$this->add_control(
			'show_arrows',
			array(
				'label'   => esc_html__( 'نمایش فلش‌ها', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'show_progress',
			array(
				'label'     => esc_html__( 'نمایش نوار پیشرفت', 'disto' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'no',
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->end_controls_section();

		// Style: container.
		$this->start_controls_section(
			'style_container',
			array(
				'label' => esc_html__( 'استایل نوار', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'container_bg',
			array(
				'label'     => esc_html__( 'رنگ پس‌زمینه', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#111827',
				'selectors' => array(
					'{{WRAPPER}} .rbn-root' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'container_radius',
			array(
				'label'      => esc_html__( 'گردی گوشه‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 40,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 8,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rbn-root' => 'border-radius: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'container_height',
			array(
				'label'      => esc_html__( 'ارتفاع نوار (px)', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 30,
						'max' => 80,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 44,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rbn-root' => 'height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'container_padding',
			array(
				'label'      => esc_html__( 'پدینگ داخلی', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px' ),
				'selectors'  => array(
					'{{WRAPPER}} .rbn-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		if ( class_exists( '\Elementor\Group_Control_Box_Shadow' ) ) {
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				array(
					'name'     => 'container_shadow',
					'selector' => '{{WRAPPER}} .rbn-root',
				)
			);
		}

		$this->end_controls_section();

		// Label.
		$this->start_controls_section(
			'style_label',
			array(
				'label' => esc_html__( 'برچسب «خبر فوری»', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'label_text',
			array(
				'label'       => esc_html__( 'متن برچسب', 'disto' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'خبر فوری', 'disto' ),
				'placeholder' => esc_html__( 'مثلاً: تازه‌ترین‌ها', 'disto' ),
			)
		);

		$this->add_control(
			'label_bg',
			array(
				'label'     => esc_html__( 'رنگ پس‌زمینه برچسب', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ef4444',
				'selectors' => array(
					'{{WRAPPER}} .rbn-label' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'label_color',
			array(
				'label'     => esc_html__( 'رنگ متن برچسب', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .rbn-label' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'label_typography',
				'label'    => esc_html__( 'تایپوگرافی برچسب', 'disto' ),
				'selector' => '{{WRAPPER}} .rbn-label',
			)
		);

		$this->add_responsive_control(
			'label_padding',
			array(
				'label'      => esc_html__( 'پدینگ برچسب', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px' ),
				'selectors'  => array(
					'{{WRAPPER}} .rbn-label' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'show_triangle',
			array(
				'label'   => esc_html__( 'نمایش مثلث کنار برچسب', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'label_triangle_color',
			array(
				'label'     => esc_html__( 'رنگ مثلث کنار برچسب', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ef4444',
				'selectors' => array(
					'{{WRAPPER}} .rbn-label-triangle' => 'border-right-color: {{VALUE}};',
				),
				'condition' => array(
					'show_triangle' => 'yes',
				),
			)
		);

		$this->end_controls_section();

		// Headline.
		$this->start_controls_section(
			'style_headline',
			array(
				'label' => esc_html__( 'استایل تیتر', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'headline_color',
			array(
				'label'     => esc_html__( 'رنگ تیتر', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .rbn-item-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'headline_hover_color',
			array(
				'label'     => esc_html__( 'رنگ هاور تیتر', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rbn-item:hover .rbn-item-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'headline_typography',
				'label'    => esc_html__( 'تایپوگرافی تیتر', 'disto' ),
				'selector' => '{{WRAPPER}} .rbn-item-title',
			)
		);

		$this->add_responsive_control(
			'label_headline_gap',
			array(
				'label'      => esc_html__( 'فاصله بین برچسب و تیتر', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 40,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 20,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rbn-headline-wrap' => 'padding-right: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'headline_align',
			array(
				'label'   => esc_html__( 'تراز متن تیتر', 'disto' ),
				'type'    => Controls_Manager::CHOOSE,
				'default' => 'right',
				'options' => array(
					'right'  => array(
						'title' => esc_html__( 'راست', 'disto' ),
						'icon'  => 'eicon-text-align-right',
					),
					'center' => array(
						'title' => esc_html__( 'وسط', 'disto' ),
						'icon'  => 'eicon-text-align-center',
					),
					'left'   => array(
						'title' => esc_html__( 'چپ', 'disto' ),
						'icon'  => 'eicon-text-align-left',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .rbn-track' => 'text-align: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		// Date style.
		$this->start_controls_section(
			'style_date',
			array(
				'label' => esc_html__( 'استایل تاریخ', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'show_date' => 'yes',
				),
			)
		);

		$this->add_control(
			'date_color',
			array(
				'label'     => esc_html__( 'رنگ تاریخ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#9ca3af',
				'selectors' => array(
					'{{WRAPPER}} .rbn-item-date' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'date_typography',
				'label'    => esc_html__( 'تایپوگرافی تاریخ', 'disto' ),
				'selector' => '{{WRAPPER}} .rbn-item-date',
			)
		);

		$this->add_responsive_control(
			'date_spacing',
			array(
				'label'      => esc_html__( 'فاصله تاریخ تا تیتر', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 24,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 8,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rbn-item-date' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// Navigation arrows style.
		$this->start_controls_section(
			'style_arrows',
			array(
				'label' => esc_html__( 'فلش‌ها', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'show_arrows' => 'yes',
				),
			)
		);

		$this->add_control(
			'arrow_color',
			array(
				'label'     => esc_html__( 'رنگ فلش', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#e5e7eb',
				'selectors' => array(
					'{{WRAPPER}} .rbn-nav-btn' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'arrow_bg',
			array(
				'label'     => esc_html__( 'پس‌زمینه فلش', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0,0,0,0.25)',
				'selectors' => array(
					'{{WRAPPER}} .rbn-nav-btn' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'arrow_size',
			array(
				'label'      => esc_html__( 'اندازه فلش (px)', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 20,
						'max' => 48,
					),
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 32,
				),
				'selectors'  => array(
					'{{WRAPPER}} .rbn-nav-btn' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'arrow_hover_color',
			array(
				'label'     => esc_html__( 'رنگ هاور فلش', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .rbn-nav-btn:hover' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();

		$args = array(
			'post_type'           => 'post',
			'posts_per_page'      => isset( $settings['posts_count'] ) ? absint( $settings['posts_count'] ) : 6,
			'post_status'         => 'publish',
			'ignore_sticky_posts' => true,
		);

		if ( ! empty( $settings['category_id'] ) ) {
			$args['cat'] = (int) $settings['category_id'];
		}

		if ( isset( $settings['order_mode'] ) && 'random' === $settings['order_mode'] ) {
			$args['orderby'] = 'rand';
		} else {
			$args['orderby'] = 'date';
			$args['order']   = 'DESC';
		}

		$query = new \WP_Query( $args );

		if ( ! $query->have_posts() ) {
			return;
		}

		$items = array();

		while ( $query->have_posts() ) {
			$query->the_post();

			$post_id   = get_the_ID();
			$title     = get_the_title();
			$link      = get_permalink();
			$date      = get_the_date( 'j F Y' );
			$cat_label = '';

			if ( ! empty( $settings['show_category'] ) ) {
				$cats = get_the_category( $post_id );
				if ( ! empty( $cats ) ) {
					$cat_label = $cats[0]->name;
				}
			}

			$items[] = array(
				'id'    => $post_id,
				'title' => $title,
				'link'  => $link,
				'date'  => $date,
				'cat'   => $cat_label,
			);
		}
		wp_reset_postdata();

		if ( empty( $items ) ) {
			return;
		}

		$payload = array(
			'autoplay'      => ! empty( $settings['enable_autoplay'] ),
			'autoplaySpeed' => isset( $settings['autoplay_speed'] ) ? absint( $settings['autoplay_speed'] ) : 3500,
			'scrollMode'    => isset( $settings['scroll_mode'] ) ? sanitize_text_field( $settings['scroll_mode'] ) : 'slide',
			'pauseOnHover'  => ! empty( $settings['pause_on_hover'] ),
			'showProgress'  => ! empty( $settings['show_progress'] ) && ! empty( $settings['enable_autoplay'] ),
			'showArrows'    => ! empty( $settings['show_arrows'] ),
		);

		if ( $payload['autoplaySpeed'] < 800 ) {
			$payload['autoplaySpeed'] = 800;
		}

		$root_id = 'rbn-' . $this->get_id();

		?>
		<div id="<?php echo esc_attr( $root_id ); ?>" class="rbn-root" dir="rtl" data-rbn-settings="<?php echo esc_attr( wp_json_encode( $payload ) ); ?>">
			<div class="rbn-inner">
				<div class="rbn-label-wrap">
					<span class="rbn-label">
						<?php echo esc_html( $settings['label_text'] ?? esc_html__( 'خبر فوری', 'disto' ) ); ?>
					</span>
					<?php if ( ! empty( $settings['show_triangle'] ) ) : ?>
						<span class="rbn-label-triangle" aria-hidden="true"></span>
					<?php endif; ?>
				</div>

				<div class="rbn-headline-wrap">
					<div class="rbn-fade rbn-fade--right"></div>
					<div class="rbn-fade rbn-fade--left"></div>

					<div class="rbn-track" aria-live="polite">
						<?php foreach ( $items as $item ) : ?>
							<a
								class="rbn-item"
								<?php echo ! empty( $settings['open_new_tab'] ) ? ' target="_blank" rel="noopener noreferrer"' : ''; ?>
								href="<?php echo esc_url( $item['link'] ); ?>"
							>
								<?php if ( ! empty( $settings['show_date'] ) && ! empty( $item['date'] ) ) : ?>
									<span class="rbn-item-date"><?php echo esc_html( $item['date'] ); ?></span>
								<?php endif; ?>

								<?php if ( ! empty( $settings['show_category'] ) && ! empty( $item['cat'] ) ) : ?>
									<span class="rbn-item-cat"><?php echo esc_html( $item['cat'] ); ?></span>
								<?php endif; ?>

								<span class="rbn-item-title"><?php echo esc_html( $item['title'] ); ?></span>
							</a>
						<?php endforeach; ?>
					</div>
				</div>

				<?php if ( ! empty( $settings['show_arrows'] ) ) : ?>
					<div class="rbn-nav" aria-hidden="true">
						<button type="button" class="rbn-nav-btn rbn-prev" aria-label="<?php echo esc_attr__( 'Previous', 'disto' ); ?>">‹</button>
						<button type="button" class="rbn-nav-btn rbn-next" aria-label="<?php echo esc_attr__( 'Next', 'disto' ); ?>">›</button>
					</div>
				<?php endif; ?>
			</div>

			<?php if ( $payload['showProgress'] ) : ?>
				<div class="rbn-progress">
					<span class="rbn-progress-bar"></span>
				</div>
			<?php endif; ?>

			<div class="rbn-skeleton" aria-hidden="true"></div>
		</div>
		<?php
	}
}


