<?php
/**
 * Blog One Widget (Elementor Widget)
 *
 * ویجت حرفه‌ای برای نمایش پست‌های وبلاگ با ساختار دو ستونی:
 * - بخش اصلی: کارت‌های پست در گرید
 * - سایدبار: لیست عمودی پست‌ها
 *
 * کاملاً RTL و سازگار با زبان فارسی
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! did_action( 'elementor/loaded' ) ) {
	return;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;

class Disto_Blog_One_Widget extends Widget_Base {

	public function get_name() {
		return 'disto_blog_one';
	}

	public function get_title() {
		return esc_html__( 'وبلاگ 1', 'disto' );
	}

	public function get_icon() {
		return 'eicon-posts-grid';
	}

	public function get_categories() {
		return array( 'disto-elements' );
	}

	public function get_style_depends() {
		return array( 'disto-child-blog-one' );
	}

	public function get_script_depends() {
		return array();
	}

	protected function register_controls() {
		// ============================================
		// Content Tab - Main Posts Section
		// ============================================
		$this->start_controls_section(
			'content_main_section',
			array(
				'label' => esc_html__( 'بخش اصلی - تنظیمات محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		// Categories options
		$terms       = get_terms(
			array(
				'taxonomy'   => 'category',
				'hide_empty' => false,
			)
		);
		$cat_options = array();
		if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
			foreach ( $terms as $term ) {
				$cat_options[ (int) $term->term_id ] = $term->name;
			}
		}

		$this->add_control(
			'main_categories',
			array(
				'label'       => esc_html__( 'دسته‌بندی پست‌های اصلی', 'disto' ),
				'type'        => Controls_Manager::SELECT2,
				'options'     => $cat_options,
				'multiple'    => true,
				'label_block' => true,
				'description' => esc_html__( 'می‌توانید یک یا چند دسته انتخاب کنید. در صورت عدم انتخاب، همه دسته‌ها نمایش داده می‌شوند.', 'disto' ),
			)
		);

		$this->add_control(
			'main_order_by',
			array(
				'label'   => esc_html__( 'مرتب‌سازی پست‌های اصلی', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'latest',
				'options' => array(
					'latest'   => esc_html__( 'جدیدترین', 'disto' ),
					'oldest'   => esc_html__( 'قدیمی‌ترین', 'disto' ),
					'alphabet' => esc_html__( 'بر اساس حروف الفبا (A-Z)', 'disto' ),
					'views'    => esc_html__( 'بیشترین بازدید', 'disto' ),
					'random'   => esc_html__( 'تصادفی', 'disto' ),
				),
			)
		);

		$this->add_control(
			'main_posts_count',
			array(
				'label'   => esc_html__( 'تعداد پست‌های اصلی', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 50,
				'step'    => 1,
				'default' => 6,
			)
		);

		$this->add_control(
			'main_show_category',
			array(
				'label'   => esc_html__( 'نمایش دسته‌بندی', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'main_show_date',
			array(
				'label'   => esc_html__( 'نمایش تاریخ', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'main_show_author',
			array(
				'label'   => esc_html__( 'نمایش نویسنده', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'main_show_image',
			array(
				'label'   => esc_html__( 'نمایش تصویر شاخص', 'disto' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		// ============================================
		// Content Tab - Sidebar Section
		// ============================================
		$this->start_controls_section(
			'content_sidebar_section',
			array(
				'label' => esc_html__( 'سایدبار - تنظیمات محتوا', 'disto' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'sidebar_title',
			array(
				'label'   => esc_html__( 'عنوان سایدبار', 'disto' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Blog And News', 'disto' ),
			)
		);

		$this->add_control(
			'sidebar_posts_count',
			array(
				'label'   => esc_html__( 'تعداد پست‌های سایدبار', 'disto' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 20,
				'step'    => 1,
				'default' => 4,
			)
		);

		$this->add_control(
			'sidebar_use_same_categories',
			array(
				'label'       => esc_html__( 'استفاده از همان دسته‌بندی بخش اصلی', 'disto' ),
				'type'        => Controls_Manager::SWITCHER,
				'default'     => 'yes',
				'description' => esc_html__( 'اگر فعال باشد، از همان دسته‌بندی بخش اصلی استفاده می‌شود.', 'disto' ),
			)
		);

		$this->add_control(
			'sidebar_categories',
			array(
				'label'       => esc_html__( 'دسته‌بندی پست‌های سایدبار', 'disto' ),
				'type'        => Controls_Manager::SELECT2,
				'options'     => $cat_options,
				'multiple'    => true,
				'label_block' => true,
				'condition'   => array(
					'sidebar_use_same_categories!' => 'yes',
				),
			)
		);

		$this->add_control(
			'sidebar_order_by',
			array(
				'label'   => esc_html__( 'مرتب‌سازی پست‌های سایدبار', 'disto' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'latest',
				'options' => array(
					'latest'   => esc_html__( 'جدیدترین', 'disto' ),
					'oldest'   => esc_html__( 'قدیمی‌ترین', 'disto' ),
					'alphabet' => esc_html__( 'بر اساس حروف الفبا (A-Z)', 'disto' ),
					'views'    => esc_html__( 'بیشترین بازدید', 'disto' ),
					'random'   => esc_html__( 'تصادفی', 'disto' ),
				),
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Widget Container
		// ============================================
		$this->start_controls_section(
			'style_container_section',
			array(
				'label' => esc_html__( 'کانتینر ویجت', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			array(
				'name'     => 'container_background',
				'label'    => esc_html__( 'پس‌زمینه', 'disto' ),
				'types'    => array( 'classic', 'gradient' ),
				'selector' => '{{WRAPPER}} .blog-one-container',
			)
		);

		$this->add_responsive_control(
			'container_padding',
			array(
				'label'      => esc_html__( 'Padding', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'container_margin',
			array(
				'label'      => esc_html__( 'Margin', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'container_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-container' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'container_box_shadow',
				'label'    => esc_html__( 'Box Shadow', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-container',
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Main Posts Cards
		// ============================================
		$this->start_controls_section(
			'style_main_cards_section',
			array(
				'label' => esc_html__( 'کارت‌های پست اصلی', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		// Card Background
		$this->add_control(
			'main_card_bg',
			array(
				'label'     => esc_html__( 'رنگ پس‌زمینه کارت', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => array(
					'{{WRAPPER}} .blog-one-main .blog-one-card' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'main_card_padding',
			array(
				'label'      => esc_html__( 'Padding کارت', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-main .blog-one-card' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'main_card_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius کارت', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-main .blog-one-card' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'main_card_border',
				'label'    => esc_html__( 'Border', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-main .blog-one-card',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'main_card_box_shadow',
				'label'    => esc_html__( 'Box Shadow', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-main .blog-one-card',
			)
		);

		// Card Hover Effect
		$this->add_control(
			'main_card_hover_heading',
			array(
				'label'     => esc_html__( 'Hover Effect', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'name'     => 'main_card_hover_box_shadow',
				'label'    => esc_html__( 'Box Shadow (Hover)', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-main .blog-one-card:hover',
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Category Badge
		// ============================================
		$this->start_controls_section(
			'style_category_section',
			array(
				'label' => esc_html__( 'برچسب دسته‌بندی', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'category_text_color',
			array(
				'label'     => esc_html__( 'رنگ متن', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-card .blog-one-category' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'category_bg_color',
			array(
				'label'     => esc_html__( 'رنگ پس‌زمینه', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-card .blog-one-category' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'category_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-card .blog-one-category',
			)
		);

		$this->add_responsive_control(
			'category_padding',
			array(
				'label'      => esc_html__( 'Padding', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-card .blog-one-category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'category_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-card .blog-one-category' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Post Title
		// ============================================
		$this->start_controls_section(
			'style_title_section',
			array(
				'label' => esc_html__( 'عنوان پست', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-card .blog-one-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'title_hover_color',
			array(
				'label'     => esc_html__( 'رنگ Hover', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-card:hover .blog-one-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-card .blog-one-title',
			)
		);

		$this->add_responsive_control(
			'title_margin',
			array(
				'label'      => esc_html__( 'Margin', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-card .blog-one-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Date & Author
		// ============================================
		$this->start_controls_section(
			'style_meta_section',
			array(
				'label' => esc_html__( 'تاریخ و نویسنده', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'meta_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-card .blog-one-meta' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'meta_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-card .blog-one-meta',
			)
		);

		$this->add_responsive_control(
			'meta_spacing',
			array(
				'label'      => esc_html__( 'فاصله بین آیتم‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-card .blog-one-meta span' => 'margin-left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .blog-one-card .blog-one-meta span:last-child' => 'margin-left: 0;',
				),
			)
		);

		$this->end_controls_section();

		// ============================================
		// Style Tab - Sidebar Section
		// ============================================
		$this->start_controls_section(
			'style_sidebar_section',
			array(
				'label' => esc_html__( 'سایدبار', 'disto' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		// Sidebar Title
		$this->add_control(
			'sidebar_title_heading',
			array(
				'label'     => esc_html__( 'عنوان سایدبار', 'disto' ),
				'type'      => Controls_Manager::HEADING,
			)
		);

		$this->add_control(
			'sidebar_title_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-sidebar .blog-one-sidebar-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'sidebar_title_alignment',
			array(
				'label'   => esc_html__( 'تراز', 'disto' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => array(
					'right'  => array(
						'title' => esc_html__( 'راست', 'disto' ),
						'icon'  => 'eicon-text-align-right',
					),
					'center' => array(
						'title' => esc_html__( 'وسط', 'disto' ),
						'icon'  => 'eicon-text-align-center',
					),
					'left'   => array(
						'title' => esc_html__( 'چپ', 'disto' ),
						'icon'  => 'eicon-text-align-left',
					),
				),
				'default' => 'right',
				'toggle'  => true,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-sidebar .blog-one-sidebar-title' => 'text-align: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sidebar_title_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-sidebar .blog-one-sidebar-title',
			)
		);

		$this->add_responsive_control(
			'sidebar_title_margin',
			array(
				'label'      => esc_html__( 'فاصله پایین', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-sidebar .blog-one-sidebar-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// Sidebar List Items
		$this->add_control(
			'sidebar_item_heading',
			array(
				'label'     => esc_html__( 'آیتم‌های لیست', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_responsive_control(
			'sidebar_item_spacing',
			array(
				'label'      => esc_html__( 'فاصله بین آیتم‌ها', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-sidebar .blog-one-sidebar-item' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// Sidebar Image
		$this->add_control(
			'sidebar_image_heading',
			array(
				'label'     => esc_html__( 'تصویر', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_responsive_control(
			'sidebar_image_width',
			array(
				'label'      => esc_html__( 'عرض', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', '%' ),
				'range'      => array(
					'px' => array(
						'min' => 50,
						'max' => 300,
					),
					'%'  => array(
						'min' => 10,
						'max' => 100,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-image' => 'width: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_image_height',
			array(
				'label'      => esc_html__( 'ارتفاع', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min' => 50,
						'max' => 300,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-image' => 'height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_image_border_radius',
			array(
				'label'      => esc_html__( 'Border Radius', 'disto' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'sidebar_image_spacing',
			array(
				'label'      => esc_html__( 'فاصله از متن', 'disto' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px', 'em' ),
				'range'      => array(
					'px' => array(
						'min' => 0,
						'max' => 50,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-image' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// Sidebar Date
		$this->add_control(
			'sidebar_date_heading',
			array(
				'label'     => esc_html__( 'تاریخ', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'sidebar_date_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-date' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sidebar_date_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-date',
			)
		);

		// Sidebar Title
		$this->add_control(
			'sidebar_item_title_heading',
			array(
				'label'     => esc_html__( 'عنوان آیتم', 'disto' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'sidebar_item_title_color',
			array(
				'label'     => esc_html__( 'رنگ', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-item-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'sidebar_item_title_hover_color',
			array(
				'label'     => esc_html__( 'رنگ Hover', 'disto' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-one-sidebar-item:hover .blog-one-sidebar-item-title' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sidebar_item_title_typography',
				'label'    => esc_html__( 'تایپوگرافی', 'disto' ),
				'selector' => '{{WRAPPER}} .blog-one-sidebar-item .blog-one-sidebar-item-title',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Get posts query arguments
	 *
	 * @param array $settings Widget settings.
	 * @return array Query arguments.
	 */
	private function get_posts_query_args( $settings, $is_sidebar = false ) {
		$args = array(
			'post_type'           => 'post',
			'post_status'         => 'publish',
			'ignore_sticky_posts' => true,
			'no_found_rows'       => true,
		);

		// Posts count
		$posts_count = $is_sidebar ? absint( $settings['sidebar_posts_count'] ) : absint( $settings['main_posts_count'] );
		$args['posts_per_page'] = $posts_count > 0 ? $posts_count : ( $is_sidebar ? 4 : 6 );

		// Categories
		if ( $is_sidebar ) {
			if ( ! empty( $settings['sidebar_use_same_categories'] ) && 'yes' === $settings['sidebar_use_same_categories'] ) {
				// Use same categories as main section
				if ( ! empty( $settings['main_categories'] ) && is_array( $settings['main_categories'] ) ) {
					$args['category__in'] = array_map( 'absint', $settings['main_categories'] );
				}
			} else {
				if ( ! empty( $settings['sidebar_categories'] ) && is_array( $settings['sidebar_categories'] ) ) {
					$args['category__in'] = array_map( 'absint', $settings['sidebar_categories'] );
				}
			}
		} else {
			if ( ! empty( $settings['main_categories'] ) && is_array( $settings['main_categories'] ) ) {
				$args['category__in'] = array_map( 'absint', $settings['main_categories'] );
			}
		}

		// Order by
		$order_by = $is_sidebar ? $settings['sidebar_order_by'] : $settings['main_order_by'];

		switch ( $order_by ) {
			case 'oldest':
				$args['orderby'] = 'date';
				$args['order']   = 'ASC';
				break;
			case 'alphabet':
				$args['orderby'] = 'title';
				$args['order']   = 'ASC';
				break;
			case 'views':
				$args['orderby']  = 'meta_value_num';
				$args['meta_key'] = 'post_views_count';
				$args['order']    = 'DESC';
				break;
			case 'random':
				$args['orderby'] = 'rand';
				break;
			case 'latest':
			default:
				$args['orderby'] = 'date';
				$args['order']   = 'DESC';
				break;
		}

		return $args;
	}

	/**
	 * Render widget output
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		// Main posts query
		$main_query = new \WP_Query( $this->get_posts_query_args( $settings, false ) );

		// Sidebar posts query
		$sidebar_query = new \WP_Query( $this->get_posts_query_args( $settings, true ) );

		$widget_id = 'blog-one-' . $this->get_id();
		?>
		<div id="<?php echo esc_attr( $widget_id ); ?>" class="blog-one-container" dir="rtl">
			<div class="blog-one-wrapper">
				<!-- Main Posts Grid -->
				<div class="blog-one-main">
					<?php if ( $main_query->have_posts() ) : ?>
						<div class="blog-one-grid">
							<?php
							while ( $main_query->have_posts() ) :
								$main_query->the_post();
								$post_id = get_the_ID();
								$title   = get_the_title();
								$link    = get_permalink();
								?>
								<article class="blog-one-card">
									<a href="<?php echo esc_url( $link ); ?>" class="blog-one-card-link">
										<?php if ( ! empty( $settings['main_show_image'] ) && 'yes' === $settings['main_show_image'] ) : ?>
											<div class="blog-one-image-wrapper">
												<?php
												if ( has_post_thumbnail( $post_id ) ) {
													echo get_the_post_thumbnail( $post_id, 'large', array( 'class' => 'blog-one-image' ) );
												} else {
													?>
													<div class="blog-one-image-placeholder"></div>
													<?php
												}
												?>
												<?php if ( ! empty( $settings['main_show_category'] ) && 'yes' === $settings['main_show_category'] ) : ?>
													<?php
													$categories = get_the_category( $post_id );
													if ( ! empty( $categories ) && ! empty( $categories[0] ) ) {
														?>
														<span class="blog-one-category"><?php echo esc_html( $categories[0]->name ); ?></span>
														<?php
													}
													?>
												<?php endif; ?>
											</div>
										<?php endif; ?>
										<div class="blog-one-content">
											<h3 class="blog-one-title"><?php echo esc_html( $title ); ?></h3>
											<?php if ( ( ! empty( $settings['main_show_date'] ) && 'yes' === $settings['main_show_date'] ) || ( ! empty( $settings['main_show_author'] ) && 'yes' === $settings['main_show_author'] ) ) : ?>
												<div class="blog-one-meta">
													<?php if ( ! empty( $settings['main_show_date'] ) && 'yes' === $settings['main_show_date'] ) : ?>
														<span class="blog-one-date"><?php echo esc_html( get_the_date( 'F j, Y', $post_id ) ); ?></span>
													<?php endif; ?>
													<?php if ( ! empty( $settings['main_show_author'] ) && 'yes' === $settings['main_show_author'] ) : ?>
														<span class="blog-one-author"><?php echo esc_html( get_the_author() ); ?></span>
													<?php endif; ?>
												</div>
											<?php endif; ?>
										</div>
									</a>
								</article>
								<?php
							endwhile;
							wp_reset_postdata();
							?>
						</div>
					<?php else : ?>
						<p class="blog-one-no-posts"><?php esc_html_e( 'پستی یافت نشد.', 'disto' ); ?></p>
					<?php endif; ?>
				</div>

				<!-- Sidebar -->
				<div class="blog-one-sidebar">
					<?php if ( ! empty( $settings['sidebar_title'] ) ) : ?>
						<h2 class="blog-one-sidebar-title"><?php echo esc_html( $settings['sidebar_title'] ); ?></h2>
					<?php endif; ?>
					<?php if ( $sidebar_query->have_posts() ) : ?>
						<div class="blog-one-sidebar-list">
							<?php
							while ( $sidebar_query->have_posts() ) :
								$sidebar_query->the_post();
								$post_id = get_the_ID();
								$title   = get_the_title();
								$link    = get_permalink();
								?>
								<a href="<?php echo esc_url( $link ); ?>" class="blog-one-sidebar-item">
									<?php if ( has_post_thumbnail( $post_id ) ) : ?>
										<div class="blog-one-sidebar-image">
											<?php echo get_the_post_thumbnail( $post_id, 'thumbnail', array( 'class' => 'blog-one-sidebar-img' ) ); ?>
										</div>
									<?php endif; ?>
									<div class="blog-one-sidebar-content">
										<span class="blog-one-sidebar-date"><?php echo esc_html( get_the_date( 'F j, Y', $post_id ) ); ?></span>
										<h4 class="blog-one-sidebar-item-title"><?php echo esc_html( $title ); ?></h4>
									</div>
								</a>
								<?php
							endwhile;
							wp_reset_postdata();
							?>
						</div>
					<?php else : ?>
						<p class="blog-one-no-posts"><?php esc_html_e( 'پستی یافت نشد.', 'disto' ); ?></p>
					<?php endif; ?>
				</div>
			</div>
		</div>
		<?php
	}
}



