<?php
/**
 * Persian digits (۰-۹) for frontend visual output.
 *
 * - Does NOT change stored values in DB
 * - Avoids scripts/styles/JSON-LD by only converting text nodes in HTML output
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

function disto_child_to_persian_digits( string $text ): string {
	static $map = array(
		'0' => '۰',
		'1' => '۱',
		'2' => '۲',
		'3' => '۳',
		'4' => '۴',
		'5' => '۵',
		'6' => '۶',
		'7' => '۷',
		'8' => '۸',
		'9' => '۹',
	);
	return strtr( $text, $map );
}

function disto_child_is_safe_context_for_digit_conversion(): bool {
	if ( is_admin() ) {
		return false;
	}
	if ( function_exists( 'wp_is_json_request' ) && wp_is_json_request() ) {
		return false;
	}
	if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
		return false;
	}
	if ( is_feed() ) {
		return false;
	}
	return true;
}

/**
 * Convert digits only in text segments (not inside tags).
 */
function disto_child_persianize_html( string $html ): string {
	if ( $html === '' ) {
		return $html;
	}

	// Split into tags and text chunks.
	if ( ! function_exists( 'wp_html_split' ) ) {
		return disto_child_to_persian_digits( $html );
	}

	$chunks = wp_html_split( $html );
	if ( ! is_array( $chunks ) ) {
		return disto_child_to_persian_digits( $html );
	}

	$out              = '';
	$skip_tag_depth   = 0;
	$skip_stack       = array();
	$skip_tag_pattern = '/^<(\/?)(script|style|textarea|pre|code)\b/i';

	foreach ( $chunks as $chunk ) {
		if ( $chunk === '' ) {
			continue;
		}
		if ( $chunk[0] === '<' ) {
			// Track tag depth for script/style/pre/code blocks.
			if ( preg_match( $skip_tag_pattern, $chunk, $m ) ) {
				$is_close = $m[1] === '/';
				$tag      = strtolower( $m[2] );

				if ( ! $is_close ) {
					// Special-case JSON-LD: never touch its contents.
					if ( $tag === 'script' && stripos( $chunk, 'application/ld+json' ) !== false ) {
						$skip_stack[] = 'script-jsonld';
						$skip_tag_depth++;
					} else {
						$skip_stack[] = $tag;
						$skip_tag_depth++;
					}
				} else {
					if ( $skip_tag_depth > 0 ) {
						array_pop( $skip_stack );
						$skip_tag_depth--;
					}
				}
			}

			$out .= $chunk;
			continue;
		}

		// Text chunk.
		if ( $skip_tag_depth > 0 ) {
			$out .= $chunk;
		} else {
			$out .= disto_child_to_persian_digits( $chunk );
		}
	}

	return $out;
}

/**
 * Apply Persian digits to common frontend outputs.
 */
add_filter( 'the_title', function ( $title ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $title ) : $title;
}, 30 );

add_filter( 'the_content', function ( $content ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $content ) : $content;
}, 30 );

add_filter( 'widget_text', function ( $text ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $text ) : $text;
}, 30 );

add_filter( 'wp_nav_menu_items', function ( $items ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $items ) : $items;
}, 30 );

// Pagination HTML.
add_filter( 'paginate_links_output', function ( $r ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $r ) : $r;
}, 30 );

// WooCommerce prices and order numbers (visual only).
add_filter( 'woocommerce_get_price_html', function ( $price_html ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $price_html ) : $price_html;
}, 30 );

add_filter( 'woocommerce_cart_item_price', function ( $price_html ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $price_html ) : $price_html;
}, 30 );

add_filter( 'woocommerce_cart_item_subtotal', function ( $subtotal_html ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_persianize_html( (string) $subtotal_html ) : $subtotal_html;
}, 30 );

add_filter( 'woocommerce_order_number', function ( $order_number ) {
	return disto_child_is_safe_context_for_digit_conversion() ? disto_child_to_persian_digits( (string) $order_number ) : $order_number;
}, 30 );

