<?php
/**
 * Disto Child theme bootstrap.
 *
 * Requirements from client:
 * - Fully Persian (fa_IR) + RTL
 * - Kalameh font loaded from theme or custom font directory (dynamic path)
 * - No Google Fonts
 * - Update-safe: no parent core edits
 * - Production-ready, clean code
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'DISTO_CHILD_VERSION', '1.0.0' );

// اسلاگ صفحه اصلی گارد لایسنس (همان صفحه‌ای که لینک آن
// wp-admin/admin.php?page=e4c683b8f215cb143256ca2c7d8e است)
if ( ! defined( 'DISTO_LICENSE_PAGE_SLUG' ) ) {
	define( 'DISTO_LICENSE_PAGE_SLUG', 'e4c683b8f215cb143256ca2c7d8e' );
}

require_once __DIR__ . '/includes/font-css-endpoint.php';
require_once __DIR__ . '/includes/jalali.php';
require_once __DIR__ . '/includes/digits.php';

/**
 * Force site locale to Persian (fa_IR).
 *
 * Note: This is theme-level enforcement. If you prefer setting it from WP Admin,
 * define `DISTO_CHILD_DISABLE_FORCE_LOCALE` in wp-config.php as true.
 */
function disto_child_force_locale( $locale ) {
	if ( defined( 'DISTO_CHILD_DISABLE_FORCE_LOCALE' ) && DISTO_CHILD_DISABLE_FORCE_LOCALE ) {
		return $locale;
	}
	return 'fa_IR';
}
add_filter( 'locale', 'disto_child_force_locale', 1 );
add_filter( 'determine_locale', 'disto_child_force_locale', 1 );
add_filter( 'pre_option_WPLANG', function ( $value ) {
	if ( defined( 'DISTO_CHILD_DISABLE_FORCE_LOCALE' ) && DISTO_CHILD_DISABLE_FORCE_LOCALE ) {
		return $value;
	}
	return 'fa_IR';
}, 1 );

/**
 * Force <html dir="rtl" lang="fa-IR"> via language_attributes filter.
 */
add_filter( 'language_attributes', function ( $output ) {
	// Remove existing dir/lang attributes (if any).
	$output = preg_replace( '/\sdir=(["\']).*?\1/i', '', $output );
	$output = preg_replace( '/\slang=(["\']).*?\1/i', '', $output );
	$output = trim( $output );

	return trim( $output . ' dir="rtl" lang="fa-IR"' );
}, 99 );

/**
 * Load child theme translations (and allow overriding parent 'disto' translations).
 */
add_action( 'after_setup_theme', function () {
	load_child_theme_textdomain( 'disto-child', get_stylesheet_directory() . '/languages' );

	// If a custom Disto translation exists in child theme, load it.
	$custom_disto_mo = get_stylesheet_directory() . '/languages/disto-fa_IR.mo';
	if ( file_exists( $custom_disto_mo ) ) {
		load_textdomain( 'disto', $custom_disto_mo );
	}

	// If a custom WooCommerce translation exists in child theme, load it.
	$custom_wc_mo = get_stylesheet_directory() . '/languages/woocommerce-fa_IR.mo';
	if ( file_exists( $custom_wc_mo ) ) {
		load_textdomain( 'woocommerce', $custom_wc_mo );
	}
}, 20 );

/**
 * Enqueue styles/scripts (RTL + typography + select2 i18n).
 */
add_action( 'wp_enqueue_scripts', function () {
	// Parent theme already enqueues its own styles. We still enqueue child style for metadata and overrides.
	wp_enqueue_style(
		'disto-child-style',
		get_stylesheet_uri(),
		array( 'disto_style' ),
		DISTO_CHILD_VERSION
	);

	// Parent theme RTL rules (shipped with Disto).
	wp_enqueue_style(
		'disto-parent-rtl',
		get_template_directory_uri() . '/rtl.css',
		array( 'disto_style' ),
		DISTO_CHILD_VERSION
	);

	// Base RTL rules + RTL fixes for Disto/Elementor/Woo.
	wp_enqueue_style(
		'disto-child-rtl',
		get_stylesheet_directory_uri() . '/rtl.css',
		array( 'disto-child-style', 'disto-parent-rtl' ),
		DISTO_CHILD_VERSION
	);

	wp_enqueue_style(
		'disto-child-rtl-fixes',
		get_stylesheet_directory_uri() . '/assets/css/rtl-fixes.css',
		array( 'disto-child-rtl' ),
		DISTO_CHILD_VERSION
	);

	// Iranian fonts (Vazir primary, IRANYekan fallback) – static CSS.
	wp_enqueue_style(
		'disto-child-fonts-fa',
		get_stylesheet_directory_uri() . '/assets/css/fonts-fa.css',
		array(),
		DISTO_CHILD_VERSION
	);

	// Global typography override (loaded after fonts).
	wp_enqueue_style(
		'disto-child-typography',
		get_stylesheet_directory_uri() . '/assets/css/typography.css',
		array( 'disto-child-fonts-fa', 'disto-child-rtl-fixes' ),
		DISTO_CHILD_VERSION
	);

	// صفحه ۴۰۴ – طراحی پیش‌فرض (فقط وقتی صفحه سفارشی انتخاب نشده).
	if ( is_404() && ! disto_child_get_custom_404_page_id() ) {
		wp_enqueue_style(
			'disto-child-404',
			get_stylesheet_directory_uri() . '/assets/css/404.css',
			array( 'disto-child-typography' ),
			DISTO_CHILD_VERSION
		);
	}

	// Select2/SelectWoo Persian + RTL (frontend only).
	wp_enqueue_script(
		'disto-child-select2-fa',
		get_stylesheet_directory_uri() . '/assets/js/select2-fa.js',
		array( 'jquery' ),
		DISTO_CHILD_VERSION,
		true
	);

	// Menu header handler - مدیریت تغییر هدر بر اساس آیتم منو
	wp_enqueue_script(
		'disto-child-menu-header-handler',
		get_stylesheet_directory_uri() . '/assets/js/menu-header-handler.js',
		array(),
		DISTO_CHILD_VERSION,
		true
	);
}, 999 );

/**
 * Remove Google Fonts (theme + Elementor) and any font links.
 */
add_action( 'wp_enqueue_scripts', function () {
	// Parent theme Google Fonts.
	wp_dequeue_style( 'disto_fonts_url' );
	wp_deregister_style( 'disto_fonts_url' );
}, 1000 );

// Elementor: prevent printing Google Fonts.
add_filter( 'elementor/frontend/print_google_fonts', '__return_false', 999 );
add_filter( 'elementor/frontend/print_font_links', '__return_false', 999 );

/**
 * Remove parent inline script (keep pages clean, avoid conflicts).
 */
add_action( 'after_setup_theme', function () {
	if ( function_exists( 'disto_custom_inline_script' ) ) {
		remove_action( 'wp_footer', 'disto_custom_inline_script' );
	}
}, 50 );

/**
 * Minimal string overrides (safety net). Full translation should be provided via .mo files.
 */
add_filter( 'gettext', function ( $translated, $text, $domain ) {
	if ( $domain !== 'disto' ) {
		return $translated;
	}

	$map = array(
		'Read More'                           => 'ادامه مطلب',
		'Load More'                           => 'بارگذاری بیشتر',
		'No more articles'                    => 'مطلب بیشتری وجود ندارد',
		'No result found'                     => 'نتیجه‌ای یافت نشد',
		'Back to Home'                        => 'بازگشت به صفحه اصلی',
		'Previous post'                       => 'نوشته قبلی',
		'Next post'                           => 'نوشته بعدی',
		'Related Articles'                    => 'مطالب مرتبط',
		'Share'                               => 'اشتراک‌گذاری',
		'Search for: '                        => 'جستجو برای: ',
		'Please try again with some different keywords.' => 'لطفاً با کلمات کلیدی دیگری دوباره تلاش کنید.',
	);

	return $map[ $text ] ?? $translated;
}, 20, 3 );

/**
 * اجبار فونت‌های تنظیمات سفارشی‌ساز (Customizer) به وزیر،
 * تا اگر قبلاً روی Philosopher یا هر فونت دیگری تنظیم شده‌اند،
 * نگذاریم آن فونت‌ها (مثل Philosopher) برای منو و زیرمنو/تیترها استفاده شوند.
 */
add_filter( 'theme_mod_disto_title_font_family', function () {
	return 'Vazir';
}, 20 );

add_filter( 'theme_mod_disto_p_font_family', function () {
	return 'Vazir';
}, 20 );

add_filter( 'theme_mod_disto_menu_font_family', function () {
	return 'Vazir';
}, 20 );

/**
 * به‌صورت سراسری، هر استایلی که فونت‌های گوگل (Roboto و غیره) را لود می‌کند خنثی کن.
 * این فیلتر روی تمام استایل‌های enqueue شده اعمال می‌شود؛
 * هر src که شامل fonts.googleapis.com یا fonts.gstatic.com باشد، خالی برمی‌گردد
 * و در نتیجه لینک Roboto و سایر گوگل‌فونت‌ها لود نخواهد شد.
 */
add_filter( 'style_loader_src', function ( $src, $handle ) {
	if ( ! is_string( $src ) || $src === '' ) {
		return $src;
	}

	if ( strpos( $src, 'fonts.googleapis.com' ) !== false || strpos( $src, 'fonts.gstatic.com' ) !== false ) {
		return '';
	}

	return $src;
}, 999, 2 );

/**
 * برگرداندن ID صفحهٔ 404 سفارشی (در صورت تنظیم).
 *
 * @return int 0 اگر تنظیم نشده، وگرنه ID برگه.
 */
function disto_child_get_custom_404_page_id() {
	$page_id = (int) get_theme_mod( 'disto_child_custom_404_page_id', 0 );
	if ( ! $page_id ) {
		return 0;
	}
	$page = get_post( $page_id );
	if ( ! $page || 'page' !== $page->post_type || 'publish' !== $page->post_status ) {
		return 0;
	}
	return $page_id;
}

/**
 * افزودن گزینهٔ انتخاب صفحهٔ 404 سفارشی به سفارشی‌ساز.
 */
function disto_child_customize_register_404( $wp_customize ) {
	$wp_customize->add_section(
		'disto_child_404_section',
		array(
			'title'       => __( 'صفحهٔ 404 - یافت نشد', 'disto' ),
			'description' => __( 'می‌توانید یک برگه طراحی‌شده با Elementor را به‌عنوان صفحهٔ 404 انتخاب کنید. در غیر این صورت طراحی پیش‌فرض نمایش داده می‌شود.', 'disto' ),
			'priority'    => 120,
		)
	);

	$pages    = get_pages( array( 'sort_column' => 'post_title' ) );
	$choices  = array( 0 => __( '— طراحی پیش‌فرض قالب —', 'disto' ) );
	foreach ( $pages as $page ) {
		$choices[ $page->ID ] = $page->post_title;
	}

	$wp_customize->add_setting(
		'disto_child_custom_404_page_id',
		array(
			'default'           => 0,
			'sanitize_callback' => 'absint',
		)
	);

	$wp_customize->add_control(
		'disto_child_custom_404_page_id',
		array(
			'label'   => __( 'صفحهٔ 404 سفارشی', 'disto' ),
			'section' => 'disto_child_404_section',
			'type'    => 'select',
			'choices' => $choices,
		)
	);
}
add_action( 'customize_register', 'disto_child_customize_register_404' );

/**
 * اطمینان از لود استایل/اسکریپت Elementor برای صفحهٔ 404 سفارشی.
 */
function disto_child_enqueue_elementor_404_assets() {
	if ( ! is_404() ) {
		return;
	}
	$page_id = disto_child_get_custom_404_page_id();
	if ( ! $page_id || ! class_exists( '\Elementor\Plugin' ) ) {
		return;
	}
	$plugin = \Elementor\Plugin::$instance;
	if ( $plugin->db->is_built_with_elementor( $page_id ) && method_exists( $plugin->frontend, 'enqueue_styles' ) ) {
		$plugin->frontend->enqueue_styles( $page_id );
	}
	if ( method_exists( $plugin->frontend, 'enqueue_scripts' ) ) {
		$plugin->frontend->enqueue_scripts( $page_id );
	}
}
add_action( 'wp_enqueue_scripts', 'disto_child_enqueue_elementor_404_assets', 15 );

/**
 * ================================
 *  Disto Theme Settings – Admin Menu
 *  نمایش منوی «تنظیمات قالب دیستو» در پیشخوان
 *  (چه وقتی خود disto فعال باشد، چه وقتی disto-child فعال باشد)
 * ================================
 */

/**
 * ثبت منوی تنظیمات قالب دیستو در پیشخوان.
 */
function disto_child_register_theme_menu() {
	// فقط اگر پوسته والد همان disto است.
	if ( get_template() !== 'disto' ) {
		return;
	}

	$capability = 'manage_options';

	// منوی اصلی تنظیمات قالب دیستو
	add_menu_page(
		__( 'تنظیمات قالب دیستو', 'disto' ),  // عنوان صفحه
		__( 'تنظیمات قالب دیستو', 'disto' ),  // عنوان در منوی سایدبار
		$capability,                           // سطح دسترسی
		'disto-theme-settings',                // slug
		'disto_child_render_theme_settings_page', // تابع رندر صفحه
		'dashicons-admin-customizer',          // آیکون
		26                                     // موقعیت تقریبی بعد از «دیدگاه‌ها»
	);
}
add_action( 'admin_menu', 'disto_child_register_theme_menu' );

/**
 * محتوای صفحه تنظیمات (ریشه اپ React).
 */
function disto_child_render_theme_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have permission to access this page.', 'disto' ) );
	}
	?>
	<div class="wrap disto-theme-settings-wrap">
		<h1 class="screen-reader-text"><?php esc_html_e( 'تنظیمات قالب دیستو', 'disto' ); ?></h1>

		<div id="disto-theme-settings-root">
			<?php esc_html_e( 'در حال بارگذاری داشبورد تنظیمات دیستو...', 'disto' ); ?>
		</div>
	</div>
	<?php
}

/**
 * بارگذاری اسکریپت‌ها و استایل‌های داشبورد تنظیمات فقط روی همین صفحه.
 *
 * این بخش فعلاً فقط آماده است؛ وقتی باندل React را ساختی مسیرها را درست کن.
 *
 * @param string $hook_suffix
 */
function disto_child_admin_enqueue_theme_assets( $hook_suffix ) {
	// فقط روی صفحه منوی ما.
	if ( 'toplevel_page_disto-theme-settings' !== $hook_suffix ) {
		return;
	}
	// از خود چایلد تم اسکریپت و استایل را لود می‌کنیم
	$version   = DISTO_CHILD_VERSION;
	$base_uri  = get_stylesheet_directory_uri() . '/assets';
	$base_path = get_stylesheet_directory() . '/assets';

	// استایل ساده برای داشبورد (در صورت نیاز بعداً بسازیم)
	$css_file = '/css/disto-admin-dashboard.css';
	if ( file_exists( $base_path . $css_file ) ) {
		wp_enqueue_style(
			'disto-theme-settings',
			$base_uri . $css_file,
			array(), // می‌توان در صورت نیاز به استایل‌های دیگر وابسته کرد
			filemtime( $base_path . $css_file )
		);
	}

	// اسکریپت داشبورد (فعلاً نسخه ساده بدون React)
	$js_file = '/js/disto-admin-dashboard.js';
	if ( file_exists( $base_path . $js_file ) ) {
		wp_enqueue_script(
			'disto-theme-settings',
			$base_uri . $js_file,
			array( 'jquery' ),
			filemtime( $base_path . $js_file ),
			true
		);

		// دیتا برای جاوااسکریپت (هم‌راستا با تنظیمات سفارشی‌ساز فعلی قالب)
		global $wp_registered_sidebars;
		$widget_areas = array();
		if ( is_array( $wp_registered_sidebars ) ) {
			foreach ( $wp_registered_sidebars as $sidebar ) {
				$widget_areas[] = array(
					'id'   => $sidebar['id'],
					'name' => $sidebar['name'],
				);
			}
		}

		wp_localize_script(
			'disto-theme-settings',
			'DistoThemeSettings',
			array(
				'nonce'          => wp_create_nonce( 'disto_theme_settings_nonce' ),
				'ajaxUrl'        => admin_url( 'admin-ajax.php' ),
				'footerSettings' => array(
					'columns'        => get_theme_mod( 'footer_columns', 'footer3col' ),
					'copyright'      => get_theme_mod(
						'jl_copyright',
						'© ' . date_i18n( 'Y' ) . ' ' . get_bloginfo( 'name' )
					),
					'columnsSidebars' => array(
						'col1' => get_theme_mod( 'disto_footer_col1_sidebar', 'footer1-sidebar' ),
						'col2' => get_theme_mod( 'disto_footer_col2_sidebar', 'footer2-sidebar' ),
						'col3' => get_theme_mod( 'disto_footer_col3_sidebar', 'footer3-sidebar' ),
						'col4' => get_theme_mod( 'disto_footer_col4_sidebar', 'footer4-sidebar' ),
					),
					'widgetAreas' => $widget_areas,
				),
				'headerSettings'   => disto_child_get_header_settings_for_js(),
				'socialSettings'   => disto_child_get_social_settings_for_js(),
				'sectionsSettings' => disto_child_get_sections_settings_for_js(),
				'license'          => array(
					'active'        => \f3a39b383f7039f3591ba96e4bece71e::b828fb6639385b04bf0a33dbf(),
					// لینک مستقیم به صفحه اصلی فعال‌سازی گارد
					'activationUrl' => admin_url( 'admin.php?page=' . DISTO_LICENSE_PAGE_SLUG ),
				),
			)
		);
	}
}

/**
 * داده‌های هدر برای داشبورد (منوها، نوار بالا، CTA، مگامنو و...).
 *
 * @return array
 */
function disto_child_get_header_settings_for_js() {
	$locations = get_registered_nav_menus();
	$assigned  = get_nav_menu_locations();
	$menu_list = array();

	// دریافت لیست تمام منوهای موجود
	$all_menus = wp_get_nav_menus();
	$menus_options = array(
		array(
			'id'   => 0,
			'name' => '— بدون منو —',
		),
	);
	foreach ( $all_menus as $menu ) {
		$menus_options[] = array(
			'id'   => $menu->term_id,
			'name' => $menu->name,
		);
	}

	foreach ( $locations as $slug => $label ) {
		$menu_id   = isset( $assigned[ $slug ] ) ? (int) $assigned[ $slug ] : 0;
		$menu_obj  = $menu_id ? wp_get_nav_menu_object( $menu_id ) : null;
		$menu_list[] = array(
			'slug'   => $slug,
			'label'  => $label,
			'menuId' => $menu_id,
			'name'   => $menu_obj ? $menu_obj->name : '',
			'editUrl' => admin_url( 'nav-menus.php' ),
		);
	}

	$header_layouts = array(
		'' => 'استفاده از هدر پیش‌فرض',
		'header_magazine_personal'   => 'استایل ۱ (شخصی)',
		'header_magazine_black'      => 'استایل ۲ (تیره)',
		'header_magazine_full_screen'=> 'استایل ۳ (تمام‌صفحه)',
		'header_layout_4'            => 'استایل ۴ (پیشنهادی)',
		'header_layout_6'            => 'استایل ۵',
		'header-header_layout_6'     => 'استایل ۶',
		'header-header_layout_06'    => 'استایل ۷',
		'header-header_layout_7'     => 'استایل ۸',
	);

	// دریافت تمام آیتم‌های منو با تنظیمات هدرشان
	$menu_items_list = disto_child_get_all_menu_items_with_header_settings();

	return array(
		'headerLayout'    => get_theme_mod( 'header_layout_design', 'header_layout_4' ),
		'headerLayouts'   => $header_layouts,
		'disableTopBar'   => (bool) get_theme_mod( 'disable_top_bar', false ),
		'topBarLeftText'  => get_theme_mod( 'disto_top_bar_left_text', '' ),
		'topBarRightText' => get_theme_mod( 'disto_top_bar_right_text', '' ),
		'dateTitle'       => get_theme_mod( 'jl_date_title', __( 'Current Date:', 'disto' ) ),
		'disableDate'     => (bool) get_theme_mod( 'jl_disable_c_date', false ),
		'stickyMenu'      => (int) get_theme_mod( 'disable_sticky_menu', 0 ) !== 1,
		'disableSearch'   => (bool) get_theme_mod( 'disable_top_search', false ),
		'disableSocial'   => (bool) get_theme_mod( 'disable_social_icons', false ),
		'ctaEnabled'      => (bool) get_theme_mod( 'disto_header_cta_enabled', false ),
		'ctaLabel'        => get_theme_mod( 'disto_header_cta_label', __( 'همین حالا شروع کنید', 'disto' ) ),
		'ctaUrl'          => get_theme_mod( 'disto_header_cta_url', '#' ),
		'ctaStyle'        => get_theme_mod( 'disto_header_cta_style', 'primary' ),
		'megaMenuEnabled' => (bool) get_theme_mod( 'disto_mega_menu_enabled', false ),
		'menuLocations'   => $menu_list,
		'menuItems'       => $menu_items_list,
		'allMenus'        => $menus_options,
		'navMenusUrl'     => admin_url( 'nav-menus.php' ),
	);
}

/**
 * دریافت تمام آیتم‌های منو با تنظیمات هدرشان
 *
 * @return array
 */
function disto_child_get_all_menu_items_with_header_settings() {
	$menu_items_list = array();
	$locations = get_nav_menu_locations();

	if ( empty( $locations ) ) {
		return $menu_items_list;
	}

	// بررسی اینکه کلاس در دسترس است
	if ( ! class_exists( 'disto_Nav_Menu_Item_Custom_Fields' ) ) {
		return $menu_items_list;
	}

	$obj_jellywp_nav = new disto_Nav_Menu_Item_Custom_Fields();

	foreach ( $locations as $location => $menu_id ) {
		if ( empty( $menu_id ) ) {
			continue;
		}

		$menu_obj = wp_get_nav_menu_object( $menu_id );
		if ( ! $menu_obj ) {
			continue;
		}

		$menu_items = wp_get_nav_menu_items( $menu_id );
		if ( empty( $menu_items ) ) {
			continue;
		}

		$location_items = array();
		foreach ( $menu_items as $item ) {
			$header_layout = get_post_meta( $item->ID, $obj_jellywp_nav->get_menu_item_postmeta_key( 'menu_header_layout' ), true );
			$location_items[] = array(
				'id'           => $item->ID,
				'title'        => $item->title,
				'url'          => $item->url,
				'headerLayout' => $header_layout ? $header_layout : '',
			);
		}

		if ( ! empty( $location_items ) ) {
			$menu_items_list[] = array(
				'location' => $location,
				'locationLabel' => isset( get_registered_nav_menus()[ $location ] ) ? get_registered_nav_menus()[ $location ] : $location,
				'menuName' => $menu_obj->name,
				'items'    => $location_items,
			);
		}
	}

	return $menu_items_list;
}
add_action( 'admin_enqueue_scripts', 'disto_child_admin_enqueue_theme_assets' );

/**
 * ذخیره تنظیمات فوتر (ستون‌ها، ناحیه‌های ویجت و کپی‌رایت) از داشبورد تنظیمات دیستو.
 */
function disto_child_ajax_save_footer_settings() {
	check_ajax_referer( 'disto_theme_settings_nonce', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Unauthorized', 'disto' ) ), 403 );
	}

	$columns   = isset( $_POST['columns'] ) ? sanitize_text_field( wp_unslash( $_POST['columns'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
	$copyright = isset( $_POST['copyright'] ) ? wp_kses_post( wp_unslash( $_POST['copyright'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification

	$sidebar_col1 = isset( $_POST['sidebar_col1'] ) ? sanitize_text_field( wp_unslash( $_POST['sidebar_col1'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
	$sidebar_col2 = isset( $_POST['sidebar_col2'] ) ? sanitize_text_field( wp_unslash( $_POST['sidebar_col2'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
	$sidebar_col3 = isset( $_POST['sidebar_col3'] ) ? sanitize_text_field( wp_unslash( $_POST['sidebar_col3'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
	$sidebar_col4 = isset( $_POST['sidebar_col4'] ) ? sanitize_text_field( wp_unslash( $_POST['sidebar_col4'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification

	$allowed_columns = array( 'footer4col', 'footer3col', 'footer2col', 'footer1col', 'footer0col' );
	if ( ! in_array( $columns, $allowed_columns, true ) ) {
		$columns = 'footer3col';
	}

	if ( '' === $copyright ) {
		$copyright = '© ' . date_i18n( 'Y' ) . ' ' . get_bloginfo( 'name' );
	}

	// اعتبارسنجی ناحیه‌های ویجت
	global $wp_registered_sidebars;
	$valid_ids = array();
	if ( is_array( $wp_registered_sidebars ) ) {
		foreach ( $wp_registered_sidebars as $sidebar ) {
			$valid_ids[] = $sidebar['id'];
		}
	}
	$default_map = array(
		'footer1-sidebar',
		'footer2-sidebar',
		'footer3-sidebar',
		'footer4-sidebar',
	);

	$map = array(
		'col1' => in_array( $sidebar_col1, $valid_ids, true ) ? $sidebar_col1 : $default_map[0],
		'col2' => in_array( $sidebar_col2, $valid_ids, true ) ? $sidebar_col2 : $default_map[1],
		'col3' => in_array( $sidebar_col3, $valid_ids, true ) ? $sidebar_col3 : $default_map[2],
		'col4' => in_array( $sidebar_col4, $valid_ids, true ) ? $sidebar_col4 : $default_map[3],
	);

	set_theme_mod( 'footer_columns', $columns );
	set_theme_mod( 'jl_copyright', $copyright );
	set_theme_mod( 'disto_footer_col1_sidebar', $map['col1'] );
	set_theme_mod( 'disto_footer_col2_sidebar', $map['col2'] );
	set_theme_mod( 'disto_footer_col3_sidebar', $map['col3'] );
	set_theme_mod( 'disto_footer_col4_sidebar', $map['col4'] );

	wp_send_json_success(
		array(
			'message' => __( 'Footer settings saved.', 'disto' ),
			'data'    => array(
				'columns'   => $columns,
				'copyright' => $copyright,
				'sidebars'  => $map,
			),
		)
	);
}
add_action( 'wp_ajax_disto_save_footer_settings', 'disto_child_ajax_save_footer_settings' );

/**
 * ذخیره تنظیمات هدر از داشبورد تنظیمات دیستو.
 */
function disto_child_ajax_save_header_settings() {
	check_ajax_referer( 'disto_theme_settings_nonce', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Unauthorized', 'disto' ) ), 403 );
	}

	$allowed_layouts = array(
		'header_magazine_personal',
		'header_magazine_black',
		'header_magazine_full_screen',
		'header_layout_4',
		'header-header_layout_6',
		'header-header_layout_06',
		'header-header_layout_7',
	);

	$layout = isset( $_POST['header_layout'] ) ? sanitize_text_field( wp_unslash( $_POST['header_layout'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
	if ( ! in_array( $layout, $allowed_layouts, true ) ) {
		$layout = 'header_layout_4';
	}

	set_theme_mod( 'header_layout_design', $layout );
	set_theme_mod( 'disable_top_bar', ! empty( $_POST['top_bar_enabled'] ) ? '0' : '1' ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'disto_top_bar_left_text', isset( $_POST['top_bar_left'] ) ? sanitize_text_field( wp_unslash( $_POST['top_bar_left'] ) ) : '' ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'disto_top_bar_right_text', isset( $_POST['top_bar_right'] ) ? sanitize_text_field( wp_unslash( $_POST['top_bar_right'] ) ) : '' ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'jl_date_title', isset( $_POST['date_title'] ) ? sanitize_text_field( wp_unslash( $_POST['date_title'] ) ) : '' ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'jl_disable_c_date', ! empty( $_POST['disable_date'] ) ? true : false ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'disable_sticky_menu', empty( $_POST['sticky_menu'] ) ? '1' : '0' ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'disable_top_search', ! empty( $_POST['disable_search'] ) ? true : false ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'disable_social_icons', ! empty( $_POST['disable_social'] ) ? true : false ); // phpcs:ignore WordPress.Security.NonceVerification

	set_theme_mod( 'disto_header_cta_enabled', ! empty( $_POST['cta_enabled'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'disto_header_cta_label', isset( $_POST['cta_label'] ) ? sanitize_text_field( wp_unslash( $_POST['cta_label'] ) ) : '' ); // phpcs:ignore WordPress.Security.NonceVerification
	set_theme_mod( 'disto_header_cta_url', isset( $_POST['cta_url'] ) ? esc_url_raw( wp_unslash( $_POST['cta_url'] ) ) : '#' ); // phpcs:ignore WordPress.Security.NonceVerification
	$cta_style = isset( $_POST['cta_style'] ) ? sanitize_text_field( wp_unslash( $_POST['cta_style'] ) ) : 'primary'; // phpcs:ignore WordPress.Security.NonceVerification
	if ( in_array( $cta_style, array( 'primary', 'secondary', 'outline' ), true ) ) {
		set_theme_mod( 'disto_header_cta_style', $cta_style );
	}
	set_theme_mod( 'disto_mega_menu_enabled', ! empty( $_POST['mega_menu_enabled'] ) ); // phpcs:ignore WordPress.Security.NonceVerification

	wp_send_json_success(
		array(
			'message' => __( 'Header settings saved.', 'disto' ),
		)
	);
}
add_action( 'wp_ajax_disto_save_header_settings', 'disto_child_ajax_save_header_settings' );

/**
 * ذخیره تنظیمات هدر برای آیتم‌های منو از داشبورد تنظیمات دیستو.
 */
function disto_child_ajax_save_menu_items_header_settings() {
	check_ajax_referer( 'disto_theme_settings_nonce', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Unauthorized', 'disto' ) ), 403 );
	}

	$menu_items = isset( $_POST['menu_items'] ) ? json_decode( stripslashes( $_POST['menu_items'] ), true ) : array(); // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput

	if ( ! is_array( $menu_items ) ) {
		wp_send_json_error( array( 'message' => __( 'Invalid data format', 'disto' ) ), 400 );
	}

	$allowed_layouts = array(
		'',
		'header_magazine_personal',
		'header_magazine_black',
		'header_magazine_full_screen',
		'header_layout_4',
		'header_layout_6',
		'header-header_layout_6',
		'header-header_layout_06',
		'header-header_layout_7',
	);

	if ( ! class_exists( 'disto_Nav_Menu_Item_Custom_Fields' ) ) {
		wp_send_json_error( array( 'message' => __( 'Menu options class not found', 'disto' ) ), 500 );
	}

	$obj_jellywp_nav = new disto_Nav_Menu_Item_Custom_Fields();
	$updated_count   = 0;

	foreach ( $menu_items as $item ) {
		if ( ! isset( $item['id'] ) || ! isset( $item['headerLayout'] ) ) {
			continue;
		}

		$item_id       = (int) $item['id'];
		$header_layout = sanitize_text_field( $item['headerLayout'] );

		// بررسی اینکه آیا آیتم منو معتبر است
		$menu_item = get_post( $item_id );
		if ( ! $menu_item || $menu_item->post_type !== 'nav_menu_item' ) {
			continue;
		}

		// بررسی اینکه layout مجاز است
		if ( ! in_array( $header_layout, $allowed_layouts, true ) ) {
			$header_layout = '';
		}

		// ذخیره تنظیمات
		$meta_key = $obj_jellywp_nav->get_menu_item_postmeta_key( 'menu_header_layout' );
		if ( empty( $header_layout ) ) {
			delete_post_meta( $item_id, $meta_key );
		} else {
			update_post_meta( $item_id, $meta_key, $header_layout );
		}

		$updated_count++;
	}

	wp_send_json_success(
		array(
			'message'      => sprintf( __( '%d آیتم منو به‌روزرسانی شد.', 'disto' ), $updated_count ),
			'updatedCount' => $updated_count,
		)
	);
}
add_action( 'wp_ajax_disto_save_menu_items_header_settings', 'disto_child_ajax_save_menu_items_header_settings' );

/**
 * ذخیره انتساب منو به موقعیت‌ها از داشبورد تنظیمات دیستو.
 */
function disto_child_ajax_save_menu_locations() {
	check_ajax_referer( 'disto_theme_settings_nonce', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Unauthorized', 'disto' ) ), 403 );
	}

	$menu_locations = isset( $_POST['menu_locations'] ) ? json_decode( stripslashes( $_POST['menu_locations'] ), true ) : array(); // phpcs:ignore WordPress.Security.NonceVerification, WordPress.Security.ValidatedSanitizedInput

	if ( ! is_array( $menu_locations ) ) {
		wp_send_json_error( array( 'message' => __( 'Invalid data format', 'disto' ) ), 400 );
	}

	$registered_locations = get_registered_nav_menus();
	$new_locations       = array();
	$updated_count       = 0;

	foreach ( $menu_locations as $location_data ) {
		if ( ! isset( $location_data['location'] ) || ! isset( $location_data['menuId'] ) ) {
			continue;
		}

		$location = sanitize_text_field( $location_data['location'] );
		$menu_id  = (int) $location_data['menuId'];

		// بررسی اینکه موقعیت معتبر است
		if ( ! isset( $registered_locations[ $location ] ) ) {
			continue;
		}

		// بررسی اینکه منو معتبر است (یا 0 برای حذف منو)
		if ( $menu_id > 0 ) {
			$menu_obj = wp_get_nav_menu_object( $menu_id );
			if ( ! $menu_obj ) {
				continue;
			}
		}

		$new_locations[ $location ] = $menu_id > 0 ? $menu_id : 0;
		$updated_count++;
	}

	// ذخیره موقعیت‌های منو
	// WordPress موقعیت‌های منو را در theme_mod ذخیره می‌کند
	if ( ! empty( $new_locations ) ) {
		// دریافت موقعیت‌های فعلی
		$current_locations = get_theme_mod( 'nav_menu_locations', array() );
		
		// ادغام با موقعیت‌های جدید
		$updated_locations = array_merge( $current_locations, $new_locations );
		
		// حذف موقعیت‌هایی که منو ندارند (menu_id = 0)
		foreach ( $updated_locations as $loc => $menu_id ) {
			if ( isset( $new_locations[ $loc ] ) && $new_locations[ $loc ] === 0 ) {
				unset( $updated_locations[ $loc ] );
			}
		}
		
		// ذخیره موقعیت‌های منو
		set_theme_mod( 'nav_menu_locations', $updated_locations );
		
		// به‌روزرسانی cache برای اطمینان از نمایش فوری تغییرات
		wp_cache_delete( 'nav_menu_locations', 'theme_mods' );
		wp_cache_delete( 'alloptions', 'options' );
	}

	wp_send_json_success(
		array(
			'message'      => sprintf( __( '%d موقعیت منو به‌روزرسانی شد.', 'disto' ), $updated_count ),
			'updatedCount' => $updated_count,
			'locations'    => $new_locations,
		)
	);
}
add_action( 'wp_ajax_disto_save_menu_locations', 'disto_child_ajax_save_menu_locations' );

/**
 * دریافت تنظیمات شبکه‌های اجتماعی برای داشبورد
 *
 * @return array
 */
function disto_child_get_social_settings_for_js() {
	$social_platforms = array(
		'facebook'     => array( 'name' => 'Facebook', 'icon' => 'fa-facebook' ),
		'twitter'      => array( 'name' => 'Twitter', 'icon' => 'fa-twitter' ),
		'instagram'    => array( 'name' => 'Instagram', 'icon' => 'fa-instagram' ),
		'linkedin'     => array( 'name' => 'LinkedIn', 'icon' => 'fa-linkedin' ),
		'youtube'      => array( 'name' => 'YouTube', 'icon' => 'fa-youtube' ),
		'pinterest'    => array( 'name' => 'Pinterest', 'icon' => 'fa-pinterest-p' ),
		'google_plus'  => array( 'name' => 'Google+', 'icon' => 'fa-google-plus' ),
		'vimeo'        => array( 'name' => 'Vimeo', 'icon' => 'fa-vimeo' ),
		'behance'      => array( 'name' => 'Behance', 'icon' => 'fa-behance' ),
		'dribble'      => array( 'name' => 'Dribbble', 'icon' => 'fa-dribbble' ),
		'deviantart'   => array( 'name' => 'DeviantArt', 'icon' => 'fa-deviantart' ),
		'dropbox'      => array( 'name' => 'Dropbox', 'icon' => 'fa-dropbox' ),
		'rss'          => array( 'name' => 'RSS', 'icon' => 'fa-rss' ),
		'skype'        => array( 'name' => 'Skype', 'icon' => 'fa-skype' ),
		'stumbleupon'  => array( 'name' => 'StumbleUpon', 'icon' => 'fa-stumbleupon' ),
		'wordpress'    => array( 'name' => 'WordPress', 'icon' => 'fa-wordpress' ),
		'yahoo'        => array( 'name' => 'Yahoo', 'icon' => 'fa-yahoo' ),
		'sound_cloud'  => array( 'name' => 'SoundCloud', 'icon' => 'fa-soundcloud' ),
	);

	$social_data = array();
	foreach ( $social_platforms as $key => $platform ) {
		$url = get_theme_mod( $key, '' );
		$social_data[] = array(
			'id'    => $key,
			'name'  => $platform['name'],
			'icon'  => $platform['icon'],
			'url'   => $url,
			'active' => ! empty( $url ),
		);
	}

	return array(
		'socialTitle' => get_theme_mod( 'jl_fl_title', 'Follow us' ),
		'platforms'   => $social_data,
		'displayLocations' => array(
			'header' => ! get_theme_mod( 'disable_social_icons', false ),
			'footer' => false, // اگر در آینده بخواهید در footer هم نمایش دهید
		),
	);
}

/**
 * ذخیره تنظیمات شبکه‌های اجتماعی از داشبورد تنظیمات دیستو.
 */
function disto_child_ajax_save_social_settings() {
	check_ajax_referer( 'disto_theme_settings_nonce', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Unauthorized', 'disto' ) ), 403 );
	}

	$social_title = isset( $_POST['social_title'] ) ? sanitize_text_field( wp_unslash( $_POST['social_title'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification
	$platforms   = isset( $_POST['platforms'] ) ? json_decode( stripslashes( $_POST['platforms'] ), true ) : array(); // phpcs:ignore WordPress.Security.NonceVerification
	$display_header = isset( $_POST['display_header'] ) ? ! empty( $_POST['display_header'] ) : true; // phpcs:ignore WordPress.Security.NonceVerification

	if ( ! is_array( $platforms ) ) {
		wp_send_json_error( array( 'message' => __( 'Invalid data format', 'disto' ) ), 400 );
	}

	// لیست پلتفرم‌های مجاز
	$allowed_platforms = array(
		'facebook',
		'twitter',
		'instagram',
		'linkedin',
		'youtube',
		'pinterest',
		'google_plus',
		'vimeo',
		'behance',
		'dribble',
		'deviantart',
		'dropbox',
		'rss',
		'skype',
		'stumbleupon',
		'wordpress',
		'yahoo',
		'sound_cloud',
	);

	// ذخیره عنوان
	set_theme_mod( 'jl_fl_title', $social_title );

	// ذخیره URL هر پلتفرم
	$updated_count = 0;
	foreach ( $platforms as $platform ) {
		if ( ! isset( $platform['id'] ) || ! in_array( $platform['id'], $allowed_platforms, true ) ) {
			continue;
		}

		$platform_id = sanitize_text_field( $platform['id'] );
		$platform_url = isset( $platform['url'] ) ? esc_url_raw( $platform['url'] ) : '';

		set_theme_mod( $platform_id, $platform_url );
		$updated_count++;
	}

	// ذخیره تنظیمات نمایش
	set_theme_mod( 'disable_social_icons', ! $display_header );

	wp_send_json_success(
		array(
			'message'      => sprintf( __( 'تنظیمات شبکه‌های اجتماعی با موفقیت ذخیره شد. (%d پلتفرم)', 'disto' ), $updated_count ),
			'updatedCount' => $updated_count,
		)
	);
}
add_action( 'wp_ajax_disto_save_social_settings', 'disto_child_ajax_save_social_settings' );

/**
 * دریافت تنظیمات بخش‌ها برای داشبورد
 *
 * @return array
 */
function disto_child_get_sections_settings_for_js() {
	$sections = array(
		array(
			'id'          => 'top_bar',
			'name'        => 'نوار بالا (Top Bar)',
			'description' => 'نوار بالای هدر که شامل منوی بالا و تاریخ می‌شود',
			'enabled'     => ! get_theme_mod( 'disable_top_bar', false ),
			'order'       => 1,
		),
		array(
			'id'          => 'header',
			'name'        => 'هدر (Header)',
			'description' => 'بخش هدر شامل لوگو و منوی اصلی',
			'enabled'     => true,
			'order'       => 2,
		),
		array(
			'id'          => 'top_search',
			'name'        => 'جستجو در هدر',
			'description' => 'دکمه جستجو در هدر',
			'enabled'     => ! get_theme_mod( 'disable_top_search', false ),
			'order'       => 3,
		),
		array(
			'id'          => 'social_icons',
			'name'        => 'آیکن‌های شبکه‌های اجتماعی',
			'description' => 'نمایش آیکن‌های شبکه‌های اجتماعی در هدر',
			'enabled'     => ! get_theme_mod( 'disable_social_icons', false ),
			'order'       => 4,
		),
		array(
			'id'          => 'current_date',
			'name'        => 'تاریخ جاری',
			'description' => 'نمایش تاریخ جاری در نوار بالا',
			'enabled'     => ! get_theme_mod( 'jl_disable_c_date', false ),
			'order'       => 5,
		),
		array(
			'id'          => 'sticky_menu',
			'name'        => 'منوی چسبنده (Sticky Menu)',
			'description' => 'منوی چسبنده که هنگام اسکرول در بالای صفحه می‌ماند',
			'enabled'     => ! get_theme_mod( 'disable_sticky_menu', false ),
			'order'       => 6,
		),
		array(
			'id'          => 'content',
			'name'        => 'محتوای اصلی',
			'description' => 'بخش محتوای اصلی سایت',
			'enabled'     => true,
			'order'       => 7,
		),
		array(
			'id'          => 'footer',
			'name'        => 'فوتر (Footer)',
			'description' => 'بخش فوتر سایت',
			'enabled'     => get_theme_mod( 'footer_columns', 'footer3col' ) !== 'footer0col',
			'order'       => 8,
		),
	);

	// دریافت ترتیب ذخیره شده (اگر وجود داشته باشد)
	$saved_order = get_theme_mod( 'disto_sections_order', array() );
	if ( ! empty( $saved_order ) && is_array( $saved_order ) ) {
		foreach ( $sections as $index => $section ) {
			if ( isset( $saved_order[ $section['id'] ] ) ) {
				$sections[ $index ]['order'] = (int) $saved_order[ $section['id'] ];
			}
		}
		// مرتب‌سازی بر اساس order
		usort( $sections, function( $a, $b ) {
			return $a['order'] - $b['order'];
		} );
	}

	return array(
		'sections' => $sections,
	);
}

/**
 * ذخیره تنظیمات بخش‌ها از داشبورد تنظیمات دیستو.
 */
function disto_child_ajax_save_sections_settings() {
	check_ajax_referer( 'disto_theme_settings_nonce', 'nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Unauthorized', 'disto' ) ), 403 );
	}

	$sections = isset( $_POST['sections'] ) ? json_decode( stripslashes( $_POST['sections'] ), true ) : array(); // phpcs:ignore WordPress.Security.NonceVerification

	if ( ! is_array( $sections ) ) {
		wp_send_json_error( array( 'message' => __( 'Invalid data format', 'disto' ) ), 400 );
	}

	// نگاشت بخش‌ها به تنظیمات theme_mod
	$section_mods = array(
		'top_bar'       => 'disable_top_bar',
		'top_search'    => 'disable_top_search',
		'social_icons'  => 'disable_social_icons',
		'current_date' => 'jl_disable_c_date',
		'sticky_menu'  => 'disable_sticky_menu',
	);

	$sections_order = array();
	$updated_count  = 0;

	foreach ( $sections as $section ) {
		if ( ! isset( $section['id'] ) || ! isset( $section['enabled'] ) || ! isset( $section['order'] ) ) {
			continue;
		}

		$section_id = sanitize_text_field( $section['id'] );
		$enabled    = ! empty( $section['enabled'] );
		$order      = (int) $section['order'];

		// ذخیره ترتیب
		$sections_order[ $section_id ] = $order;

		// ذخیره وضعیت فعال/غیرفعال
		if ( isset( $section_mods[ $section_id ] ) ) {
			$mod_key = $section_mods[ $section_id ];
			set_theme_mod( $mod_key, ! $enabled );
			$updated_count++;
		}

		// مدیریت فوتر
		if ( 'footer' === $section_id ) {
			if ( ! $enabled ) {
				set_theme_mod( 'footer_columns', 'footer0col' );
			} else {
				// اگر فوتر غیرفعال بود و حالا فعال شد، به پیش‌فرض برگردان
				$current_footer = get_theme_mod( 'footer_columns', 'footer3col' );
				if ( 'footer0col' === $current_footer ) {
					set_theme_mod( 'footer_columns', 'footer3col' );
				}
			}
			$updated_count++;
		}
	}

	// ذخیره ترتیب بخش‌ها
	set_theme_mod( 'disto_sections_order', $sections_order );

	wp_send_json_success(
		array(
			'message'      => sprintf( __( 'تنظیمات بخش‌ها با موفقیت ذخیره شد. (%d بخش)', 'disto' ), $updated_count ),
			'updatedCount' => $updated_count,
		)
	);
}
add_action( 'wp_ajax_disto_save_sections_settings', 'disto_child_ajax_save_sections_settings' );

/**
 * تشخیص هدر مناسب بر اساس صفحه فعلی و آیتم منو
 * 
 * این تابع بررسی می‌کند که آیا صفحه فعلی از طریق یک آیتم منو که هدر خاصی دارد باز شده است یا نه
 * 
 * @return string|null هدر مناسب یا null اگر هدر پیش‌فرض باید استفاده شود
 */
function disto_child_get_header_layout_for_current_page() {
	// لیست هدرهای مجاز
	$allowed_layouts = array(
		'header_magazine_personal',
		'header_magazine_black',
		'header_magazine_full_screen',
		'header_layout_4',
		'header_layout_6',
		'header-header_layout_6',
		'header-header_layout_06',
		'header-header_layout_7',
	);

	// اولویت 1: بررسی query parameter (از JavaScript اضافه می‌شود)
	$header_from_param = isset( $_GET['header_layout'] ) ? sanitize_text_field( wp_unslash( $_GET['header_layout'] ) ) : '';
	if ( ! empty( $header_from_param ) && in_array( $header_from_param, $allowed_layouts, true ) ) {
		return $header_from_param;
	}

	// اولویت 2: بررسی بر اساس URL فعلی و منوها (برای صفحاتی که مستقیماً باز می‌شوند)
	$current_url = ( is_ssl() ? 'https://' : 'http://' ) . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
	$current_url = remove_query_arg( array( 'header_layout' ), $current_url );
	$current_url_clean = strtok( $current_url, '?' );

	$locations = get_nav_menu_locations();
	if ( ! empty( $locations ) ) {
		foreach ( $locations as $location => $menu_id ) {
			if ( empty( $menu_id ) ) {
				continue;
			}
			$menu_items = wp_get_nav_menu_items( $menu_id );
			if ( ! empty( $menu_items ) ) {
				foreach ( $menu_items as $menu_item ) {
					$menu_url = $menu_item->url;
					// حذف query parameters برای مقایسه
					$menu_url_clean = strtok( $menu_url, '?' );

					// بررسی تطابق دقیق URL
					if ( ! empty( $menu_url_clean ) && $menu_url_clean === $current_url_clean ) {
						if ( class_exists( 'disto_Nav_Menu_Item_Custom_Fields' ) ) {
							$obj_jellywp_nav = new disto_Nav_Menu_Item_Custom_Fields();
							$menu_header_layout = get_post_meta( $menu_item->ID, $obj_jellywp_nav->get_menu_item_postmeta_key( 'menu_header_layout' ), true );
							if ( ! empty( $menu_header_layout ) && in_array( $menu_header_layout, $allowed_layouts, true ) ) {
								return $menu_header_layout;
							}
						}
					}
				}
			}
		}
	}

	return null;
}

/**
 * فیلتر برای تغییر هدر بر اساس صفحه فعلی
 */
add_filter( 'theme_mod_header_layout_design', 'disto_child_filter_header_layout', 20 );
function disto_child_filter_header_layout( $default_layout ) {
	$custom_layout = disto_child_get_header_layout_for_current_page();
	if ( ! empty( $custom_layout ) ) {
		return $custom_layout;
	}
	return $default_layout;
}

/**
 * ================================
 *  Elementor – Disto widgets & category
 * ================================
 */

/**
 * Register custom Elementor category for Disto.
 *
 * This will appear in the Elementor widgets panel as "ویجت‌های دیستو".
 *
 * @param \Elementor\Elements_Manager $elements_manager Elements manager.
 */
function disto_child_register_elementor_category( $elements_manager ) {
	if ( ! did_action( 'elementor/loaded' ) ) {
		return;
	}

	$elements_manager->add_category(
		'disto-elements',
		array(
			'title' => esc_html__( 'ویجت‌های دیستو', 'disto' ),
			'icon'  => 'fa fa-plug',
		)
	);
}
add_action( 'elementor/elements/categories_registered', 'disto_child_register_elementor_category' );

/**
 * Register custom Elementor widgets for Disto category.
 *
 * @param \Elementor\Widgets_Manager $widgets_manager Widgets manager.
 */
function disto_child_register_elementor_widgets( $widgets_manager ) {
	if ( ! did_action( 'elementor/loaded' ) ) {
		return;
	}

	$rnc_widget_file    = get_stylesheet_directory() . '/includes/elementor/class-rtl-news-carousel-pro-widget.php';
	$rnc_cards_widget   = get_stylesheet_directory() . '/includes/elementor/class-rtl-news-carousel-cards-widget.php';
	$rbn_widget_file     = get_stylesheet_directory() . '/includes/elementor/class-breaking-news-ticker-pro-widget.php';
	$about_us_widget     = get_stylesheet_directory() . '/includes/elementor/class-about-us-widget.php';
	$blog_one_widget    = get_stylesheet_directory() . '/includes/elementor/class-blog-one-widget.php';
	$news_sidebar_widget = get_stylesheet_directory() . '/includes/elementor/class-news-sidebar-widget.php';
	$woocommerce_products_widget = get_stylesheet_directory() . '/includes/elementor/class-woocommerce-products-widget.php';
	$pro_portfolio_widget        = get_stylesheet_directory() . '/includes/elementor/class-pro-portfolio-widget.php';
	$faq_widget                  = get_stylesheet_directory() . '/includes/elementor/class-faq-widget.php';
	$team_met_pro_widget         = get_stylesheet_directory() . '/includes/elementor/class-team-met-pro-widget.php';
	$coming_soon_widget          = get_stylesheet_directory() . '/includes/elementor/class-coming-soon-widget.php';
	$team_box_widget             = get_stylesheet_directory() . '/includes/elementor/class-disto-team-box-widget.php';

	if ( file_exists( $rnc_widget_file ) ) {
		require_once $rnc_widget_file;
	}
	if ( file_exists( $rnc_cards_widget ) ) {
		require_once $rnc_cards_widget;
	}
	if ( file_exists( $rbn_widget_file ) ) {
		require_once $rbn_widget_file;
	}
	if ( file_exists( $about_us_widget ) ) {
		require_once $about_us_widget;
	}
	if ( file_exists( $blog_one_widget ) ) {
		require_once $blog_one_widget;
	}
	if ( file_exists( $news_sidebar_widget ) ) {
		require_once $news_sidebar_widget;
	}
	if ( file_exists( $woocommerce_products_widget ) && class_exists( 'WooCommerce' ) ) {
		require_once $woocommerce_products_widget;
	}
	if ( file_exists( $pro_portfolio_widget ) ) {
		require_once $pro_portfolio_widget;
	}
	if ( file_exists( $faq_widget ) ) {
		require_once $faq_widget;
	}
	if ( file_exists( $team_met_pro_widget ) ) {
		require_once $team_met_pro_widget;
	}
	if ( file_exists( $coming_soon_widget ) ) {
		require_once $coming_soon_widget;
	}
	if ( file_exists( $team_box_widget ) ) {
		require_once $team_box_widget;
	}

	if ( class_exists( 'Disto_RTL_News_Carousel_Pro_Widget' ) ) {
		$widgets_manager->register( new \Disto_RTL_News_Carousel_Pro_Widget() );
	}
	if ( class_exists( 'Disto_RTL_News_Carousel_Cards_Widget' ) ) {
		$widgets_manager->register( new \Disto_RTL_News_Carousel_Cards_Widget() );
	}
	if ( class_exists( 'Disto_Breaking_News_Ticker_Pro_Widget' ) ) {
		$widgets_manager->register( new \Disto_Breaking_News_Ticker_Pro_Widget() );
	}
	if ( class_exists( 'Disto_About_Us_Section_Widget' ) ) {
		$widgets_manager->register( new \Disto_About_Us_Section_Widget() );
	}
	if ( class_exists( 'Disto_Blog_One_Widget' ) ) {
		$widgets_manager->register( new \Disto_Blog_One_Widget() );
	}
	if ( class_exists( 'Disto_News_Sidebar_Widget' ) ) {
		$widgets_manager->register( new \Disto_News_Sidebar_Widget() );
	}
	if ( class_exists( 'Disto_WooCommerce_Products_Widget' ) && class_exists( 'WooCommerce' ) ) {
		$widgets_manager->register( new \Disto_WooCommerce_Products_Widget() );
	}
	if ( class_exists( 'Disto_Pro_Portfolio_Widget' ) ) {
		$widgets_manager->register( new \Disto_Pro_Portfolio_Widget() );
	}
	if ( class_exists( 'Disto_FAQ_Widget' ) ) {
		$widgets_manager->register( new \Disto_FAQ_Widget() );
	}
	if ( class_exists( 'Disto_Team_Met_Pro_Widget' ) ) {
		$widgets_manager->register( new \Disto_Team_Met_Pro_Widget() );
	}
	if ( class_exists( 'Disto_Coming_Soon_Widget' ) ) {
		$widgets_manager->register( new \Disto_Coming_Soon_Widget() );
	}
	if ( class_exists( 'Disto_Team_Box_Widget' ) ) {
		$widgets_manager->register( new \Disto_Team_Box_Widget() );
	}
}
add_action( 'elementor/widgets/register', 'disto_child_register_elementor_widgets' );

/**
 * Register widget assets for Elementor widgets.
 */
function disto_child_register_rnc_assets() {
	// Swiper from CDN (no jQuery).
	if ( ! wp_style_is( 'disto-child-swiper', 'registered' ) ) {
		wp_register_style(
			'disto-child-swiper',
			'https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.css',
			array(),
			'11.0.0'
		);
	}

	if ( ! wp_script_is( 'disto-child-swiper', 'registered' ) ) {
		wp_register_script(
			'disto-child-swiper',
			'https://cdn.jsdelivr.net/npm/swiper@11/swiper-bundle.min.js',
			array(),
			'11.0.0',
			true
		);
	}

	$css_rel = '/assets/css/rnc-rtl-news-carousel-pro.css';
	$js_rel  = '/assets/js/rnc-rtl-news-carousel-pro.js';
	$about_css = '/assets/css/about-us-widget.css';

	if ( ! wp_style_is( 'disto-child-rnc', 'registered' ) ) {
		wp_register_style(
			'disto-child-rnc',
			get_stylesheet_directory_uri() . $css_rel,
			array( 'disto-child-swiper' ),
			DISTO_CHILD_VERSION
		);
	}

	if ( ! wp_script_is( 'disto-child-rnc', 'registered' ) ) {
		wp_register_script(
			'disto-child-rnc',
			get_stylesheet_directory_uri() . $js_rel,
			array( 'disto-child-swiper' ),
			DISTO_CHILD_VERSION,
			true
		);

		wp_localize_script(
			'disto-child-rnc',
			'RNC_GLOBALS',
			array(
				'restUrl'   => esc_url_raw( rest_url( 'wp/v2/posts' ) ),
				'restNonce' => wp_create_nonce( 'wp_rest' ),
			)
		);
	}

	// About Us widget styles.
	if ( ! wp_style_is( 'disto-child-about-us', 'registered' ) ) {
		wp_register_style(
			'disto-child-about-us',
			get_stylesheet_directory_uri() . $about_css,
			array(),
			DISTO_CHILD_VERSION
		);
	}

	// Breaking News Ticker assets (no external dependencies).
	$rbn_css = '/assets/css/rbn-breaking-news-ticker.css';
	$rbn_js  = '/assets/js/rbn-breaking-news-ticker.js';

	if ( ! wp_style_is( 'disto-child-rbn', 'registered' ) ) {
		wp_register_style(
			'disto-child-rbn',
			get_stylesheet_directory_uri() . $rbn_css,
			array(),
			DISTO_CHILD_VERSION
		);
	}

	if ( ! wp_script_is( 'disto-child-rbn', 'registered' ) ) {
		wp_register_script(
			'disto-child-rbn',
			get_stylesheet_directory_uri() . $rbn_js,
			array(),
			DISTO_CHILD_VERSION,
			true
		);
	}

	// استایل گرید «پست شبکه‌ای»
	if ( ! wp_style_is( 'disto-child-news-cards', 'registered' ) ) {
		wp_register_style(
			'disto-child-news-cards',
			get_stylesheet_directory_uri() . '/assets/css/disto-news-card.css',
			array(),
			DISTO_CHILD_VERSION
		);
	}

	// Blog One widget styles.
	if ( ! wp_style_is( 'disto-child-blog-one', 'registered' ) ) {
		wp_register_style(
			'disto-child-blog-one',
			get_stylesheet_directory_uri() . '/assets/css/blog-one-widget.css',
			array(),
			DISTO_CHILD_VERSION
		);
	}

	// News Sidebar widget styles.
	if ( ! wp_style_is( 'disto-child-news-sidebar', 'registered' ) ) {
		wp_register_style(
			'disto-child-news-sidebar',
			get_stylesheet_directory_uri() . '/assets/css/news-sidebar-widget.css',
			array(),
			DISTO_CHILD_VERSION
		);
	}

	// WooCommerce Products widget styles.
	if ( ! wp_style_is( 'disto-child-woocommerce-products', 'registered' ) && class_exists( 'WooCommerce' ) ) {
		wp_register_style(
			'disto-child-woocommerce-products',
			get_stylesheet_directory_uri() . '/assets/css/woocommerce-products-widget.css',
			array(),
			DISTO_CHILD_VERSION
		);
	}

	// WooCommerce Products widget JavaScript.
	if ( ! wp_script_is( 'disto-child-woocommerce-products', 'registered' ) && class_exists( 'WooCommerce' ) ) {
		wp_register_script(
			'disto-child-woocommerce-products',
			get_stylesheet_directory_uri() . '/assets/js/woocommerce-products-widget.js',
			array( 'jquery', 'wc-add-to-cart' ),
			DISTO_CHILD_VERSION,
			true
		);

		wp_localize_script(
			'disto-child-woocommerce-products',
			'DWP_AJAX',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'dwp_add_to_cart_nonce' ),
			)
		);
	}

	// Pro Portfolio widget.
	if ( ! wp_style_is( 'disto-child-pro-portfolio', 'registered' ) ) {
		wp_register_style(
			'disto-child-pro-portfolio',
			get_stylesheet_directory_uri() . '/assets/css/pro-portfolio-widget.css',
			array( 'disto-child-swiper' ),
			DISTO_CHILD_VERSION
		);
	}
	if ( ! wp_script_is( 'disto-child-pro-portfolio', 'registered' ) ) {
		wp_register_script(
			'disto-child-pro-portfolio',
			get_stylesheet_directory_uri() . '/assets/js/pro-portfolio-widget.js',
			array( 'disto-child-swiper' ),
			DISTO_CHILD_VERSION,
			true
		);
	}

	// FAQ (سوالات متداول) widget.
	if ( ! wp_style_is( 'disto-child-faq-widget', 'registered' ) ) {
		wp_register_style(
			'disto-child-faq-widget',
			get_stylesheet_directory_uri() . '/assets/css/faq-widget.css',
			array(),
			DISTO_CHILD_VERSION
		);
	}
	if ( ! wp_script_is( 'disto-child-faq-widget', 'registered' ) ) {
		wp_register_script(
			'disto-child-faq-widget',
			get_stylesheet_directory_uri() . '/assets/js/faq-widget.js',
			array(),
			DISTO_CHILD_VERSION,
			true
		);
	}

	// Team Met Pro widget.
	if ( ! wp_style_is( 'disto-child-team-met-pro', 'registered' ) ) {
		wp_register_style(
			'disto-child-team-met-pro',
			get_stylesheet_directory_uri() . '/assets/css/team-met-pro-widget.css',
			array( 'disto-child-swiper' ),
			DISTO_CHILD_VERSION
		);
	}
	if ( ! wp_script_is( 'disto-child-team-met-pro', 'registered' ) ) {
		wp_register_script(
			'disto-child-team-met-pro',
			get_stylesheet_directory_uri() . '/assets/js/team-met-pro-widget.js',
			array( 'jquery', 'disto-child-swiper' ),
			DISTO_CHILD_VERSION,
			true
		);
	}

	// Team Box (ابزارک باکس تیم ما) widget.
	if ( ! wp_style_is( 'disto-child-team-box', 'registered' ) ) {
		wp_register_style(
			'disto-child-team-box',
			get_stylesheet_directory_uri() . '/assets/css/team-box-widget.css',
			array(),
			DISTO_CHILD_VERSION
		);
	}
	if ( ! wp_script_is( 'disto-child-team-box', 'registered' ) ) {
		wp_register_script(
			'disto-child-team-box',
			get_stylesheet_directory_uri() . '/assets/js/team-box-widget.js',
			array( 'jquery' ),
			DISTO_CHILD_VERSION,
			true
		);
	}

	// Coming Soon (بزودی) widget.
	if ( ! wp_style_is( 'disto-child-coming-soon', 'registered' ) ) {
		wp_register_style(
			'disto-child-coming-soon',
			get_stylesheet_directory_uri() . '/assets/css/coming-soon-widget.css',
			array(),
			DISTO_CHILD_VERSION
		);
	}
	if ( ! wp_script_is( 'disto-child-coming-soon', 'registered' ) ) {
		wp_register_script(
			'disto-child-coming-soon',
			get_stylesheet_directory_uri() . '/assets/js/coming-soon-widget.js',
			array(),
			DISTO_CHILD_VERSION,
			true
		);
	}
}
add_action( 'wp_enqueue_scripts', 'disto_child_register_rnc_assets', 20 );
add_action( 'elementor/editor/after_enqueue_scripts', 'disto_child_register_rnc_assets', 20 );
add_action( 'elementor/frontend/after_register_scripts', 'disto_child_register_rnc_assets', 20 );
add_action( 'elementor/frontend/after_register_styles', 'disto_child_register_rnc_assets', 20 );

/**
 * AJAX handler for adding product to cart from WooCommerce Products Widget
 */
function disto_child_ajax_add_to_cart() {
	check_ajax_referer( 'dwp_add_to_cart_nonce', 'nonce' );

	if ( ! class_exists( 'WooCommerce' ) ) {
		wp_send_json_error( array( 'message' => esc_html__( 'WooCommerce is not active.', 'disto' ) ) );
	}

	$product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
	$quantity   = isset( $_POST['quantity'] ) ? absint( $_POST['quantity'] ) : 1;

	if ( ! $product_id ) {
		wp_send_json_error( array( 'message' => esc_html__( 'Product ID is required.', 'disto' ) ) );
	}

	$product = wc_get_product( $product_id );
	if ( ! $product ) {
		wp_send_json_error( array( 'message' => esc_html__( 'Product not found.', 'disto' ) ) );
	}

	// افزودن محصول به سبد خرید
	$cart_item_key = WC()->cart->add_to_cart( $product_id, $quantity );

	if ( $cart_item_key ) {
		// به‌روزرسانی fragments برای به‌روزرسانی سبد خرید در صفحه
		WC()->cart->calculate_totals();

		ob_start();
		woocommerce_mini_cart();
		$mini_cart = ob_get_clean();

		$data = array(
			'fragments' => apply_filters(
				'woocommerce_add_to_cart_fragments',
				array(
					'div.widget_shopping_cart_content' => '<div class="widget_shopping_cart_content">' . $mini_cart . '</div>',
				)
			),
			'cart_hash' => WC()->cart->get_cart_hash(),
		);

		wp_send_json_success( $data );
	} else {
		wp_send_json_error( array( 'message' => esc_html__( 'Failed to add product to cart.', 'disto' ) ) );
	}
}
add_action( 'wp_ajax_dwp_add_to_cart', 'disto_child_ajax_add_to_cart' );
add_action( 'wp_ajax_nopriv_dwp_add_to_cart', 'disto_child_ajax_add_to_cart' );

/**
 * اطمینان از سازگاری کامل با Elementor
 * این فیلتر مطمئن می‌شود که Elementor می‌تواند محتوا را به درستی نمایش دهد
 */
function disto_child_elementor_support() {
	// بررسی اینکه آیا Elementor فعال است
	if ( ! did_action( 'elementor/loaded' ) ) {
		return;
	}

	// اطمینان از اینکه the_content() در قالب‌ها فراخوانی می‌شود
	// این کار از طریق فایل‌های page.php، page-fullwidth.php و single.php انجام شده است
}
add_action( 'after_setup_theme', 'disto_child_elementor_support' );

/**
 * اطمینان از اینکه Elementor می‌تواند محتوای صفحه را به درستی نمایش دهد
 * این فیلتر برای صفحاتی که با Elementor ساخته شده‌اند استفاده می‌شود
 * فقط در صورتی اعمال می‌شود که صفحه با Elementor ساخته شده باشد
 */
function disto_child_elementor_content_support( $content ) {
	// اگر صفحه با Elementor ساخته شده باشد و محتوای Elementor وجود داشته باشد
	if ( class_exists( '\Elementor\Plugin' ) && \Elementor\Plugin::instance()->db->is_built_with_elementor( get_the_ID() ) ) {
		// Elementor خودش محتوا را مدیریت می‌کند، پس نیازی به تغییر نیست
		// فقط مطمئن می‌شویم که the_content() فراخوانی شده است
		return $content;
	}
	return $content;
}
add_filter( 'the_content', 'disto_child_elementor_content_support', 10 );
