/**
 * Team Met Pro Widget JavaScript
 * Production-ready JS with Swiper integration and animations
 */

(function($) {
	'use strict';

	/**
	 * Initialize Team Met Pro Widget
	 */
	function initTeamMetPro() {
		$('.team-met-pro-wrapper').each(function() {
			const $wrapper = $(this);
			const layoutType = $wrapper.hasClass('team-met-pro-layout-slider') ? 'slider' : 'grid';
			
			// Initialize Slider
			if (layoutType === 'slider') {
				initSlider($wrapper);
			}
			
			// با موس روی ویجت: توقف اتوپلی و غیرفعال کردن افکت/انیمیشن
			if ($wrapper.attr('data-pause-effects-on-hover') === 'yes') {
				$wrapper.off('mouseenter.teamMetPro mouseleave.teamMetPro').on('mouseenter.teamMetPro', function() {
					$wrapper.addClass('team-met-pro-mouse-over');
					const swiper = $wrapper.find('.team-met-pro-slider').data('swiper');
					if (swiper && swiper.autoplay && swiper.autoplay.stop) {
						swiper.autoplay.stop();
					}
				}).on('mouseleave.teamMetPro', function() {
					$wrapper.removeClass('team-met-pro-mouse-over');
					const swiper = $wrapper.find('.team-met-pro-slider').data('swiper');
					if (swiper && swiper.autoplay && swiper.autoplay.start) {
						swiper.autoplay.start();
					}
				});
			}
			
			// Initialize Animations
			if ($wrapper.hasClass('team-met-pro-animate-fade') || 
				$wrapper.hasClass('team-met-pro-animate-slide-up') || 
				$wrapper.hasClass('team-met-pro-animate-slide-right') || 
				$wrapper.hasClass('team-met-pro-animate-zoom-in') || 
				$wrapper.hasClass('team-met-pro-animate-flip')) {
				initAnimations($wrapper);
			}
		});
	}

	/**
	 * Initialize Swiper Slider (سریع: اگر Swiper آماده است بلافاصله اجرا)
	 */
	function initSlider($wrapper) {
		const $slider = $wrapper.find('.team-met-pro-slider');
		
		if (!$slider.length) {
			return;
		}

		function tryInit() {
			if (typeof Swiper !== 'undefined') {
				initializeSwiper($slider);
				return true;
			}
			return false;
		}

		// بلافاصله امتحان کن؛ اگر Swiper از قبل لود شده باشد تأخیری نیست
		if (tryInit()) {
			return;
		}

		// در غیر این صورت هر 50ms چک کن تا حداکثر 3 ثانیه
		let attempts = 0;
		const maxAttempts = 60;
		const checkSwiper = setInterval(function() {
			if (tryInit() || ++attempts >= maxAttempts) {
				clearInterval(checkSwiper);
			}
		}, 50);
	}

	/**
	 * Initialize Swiper instance
	 */
	function initializeSwiper($slider) {
		if (typeof Swiper === 'undefined') {
			return;
		}

		// Check if already initialized
		if ($slider.data('swiper')) {
			return;
		}

		const settingsData = $slider.data('settings');
		let settings = {};
		
		if (settingsData) {
			try {
				settings = typeof settingsData === 'string' ? JSON.parse(settingsData) : settingsData;
			} catch (e) {
				console.error('Team Met Pro: Invalid slider settings', e);
				settings = getDefaultSliderSettings();
			}
		} else {
			settings = getDefaultSliderSettings();
		}

		// Check if RTL
		if (document.documentElement.dir === 'rtl' || $('body').hasClass('rtl')) {
			settings.direction = 'rtl';
		}

		// Initialize Swiper
		try {
			const swiper = new Swiper($slider[0], settings);
			// Store swiper instance
			$slider.data('swiper', swiper);
		} catch (e) {
			console.error('Team Met Pro: Swiper initialization error', e);
		}
	}

	/**
	 * Get default slider settings
	 */
	function getDefaultSliderSettings() {
		return {
			autoplay: {
				delay: 3000,
				pauseOnMouseEnter: true,
			},
			loop: true,
			speed: 500,
			slidesPerView: 3,
			spaceBetween: 30,
			navigation: {
				nextEl: '.team-met-pro-arrow-next',
				prevEl: '.team-met-pro-arrow-prev',
			},
			pagination: {
				el: '.swiper-pagination',
				clickable: true,
			},
			breakpoints: {
				768: {
					slidesPerView: 2,
				},
				480: {
					slidesPerView: 1,
				},
			},
		};
	}

	/**
	 * Initialize Animations
	 */
	function initAnimations($wrapper) {
		const animationDuration = parseInt($wrapper.data('animation-duration')) || 800;
		const animationDelay = parseInt($wrapper.data('animation-delay')) || 100;
		const staggerAnimation = $wrapper.data('stagger') === 'yes';
		
		const $items = $wrapper.find('.team-met-pro-item-wrapper');
		
		if (!$items.length) {
			return;
		}

		// Check if element is in viewport
		if (!isInViewport($wrapper[0])) {
			// Use Intersection Observer for better performance
			if ('IntersectionObserver' in window) {
				const observer = new IntersectionObserver(function(entries) {
					entries.forEach(function(entry) {
						if (entry.isIntersecting) {
							animateItems($items, animationDelay, staggerAnimation);
							observer.unobserve(entry.target);
						}
					});
				}, {
					threshold: 0.1,
				});
				
				observer.observe($wrapper[0]);
			} else {
				// Fallback for older browsers
				$(window).on('scroll', function() {
					if (isInViewport($wrapper[0])) {
						animateItems($items, animationDelay, staggerAnimation);
						$(window).off('scroll');
					}
				});
				
				// Trigger on load if already visible
				if (isInViewport($wrapper[0])) {
					animateItems($items, animationDelay, staggerAnimation);
				}
			}
		} else {
			animateItems($items, animationDelay, staggerAnimation);
		}
	}

	/**
	 * Animate items
	 */
	function animateItems($items, delay, stagger) {
		$items.each(function(index) {
			const $item = $(this);
			const itemDelay = stagger ? delay * index : delay;
			
			setTimeout(function() {
				$item.css({
					'animation-duration': '800ms',
					'animation-fill-mode': 'forwards',
				});
			}, itemDelay);
		});
	}

	/**
	 * Check if element is in viewport
	 */
	function isInViewport(element) {
		const rect = element.getBoundingClientRect();
		return (
			rect.top >= 0 &&
			rect.left >= 0 &&
			rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) &&
			rect.right <= (window.innerWidth || document.documentElement.clientWidth)
		);
	}

	/**
	 * Handle Resize Events
	 */
	function handleResize() {
		$('.team-met-pro-slider').each(function() {
			const $slider = $(this);
			const swiper = $slider.data('swiper');
			
			if (swiper && typeof swiper.update === 'function') {
				swiper.update();
			}
		});
	}

	// Initialize on document ready
	$(document).ready(function() {
		initTeamMetPro();
	});

	// Initialize on Elementor frontend
	if (typeof elementorFrontend !== 'undefined') {
		elementorFrontend.hooks.addAction('frontend/element_ready/team_met_pro.default', function($scope) {
			initTeamMetPro();
		});
		
		// Also initialize after Swiper is loaded
		elementorFrontend.hooks.addAction('frontend/element_ready/global', function($scope) {
			if ($scope.find('.team-met-pro-slider').length) {
				setTimeout(function() {
					initTeamMetPro();
				}, 500);
			}
		});
	}

	// Handle window resize
	let resizeTimer;
	$(window).on('resize', function() {
		clearTimeout(resizeTimer);
		resizeTimer = setTimeout(function() {
			handleResize();
		}, 250);
	});

	// Reinitialize on AJAX content load
	$(document).on('elementor/popup/show', function() {
		setTimeout(function() {
			initTeamMetPro();
		}, 300);
	});

})(jQuery);
