/* global RNC_GLOBALS, elementorFrontend */
(function () {
  'use strict';

  function stripHtml(html) {
    if (!html) return '';
    const tmp = document.createElement('div');
    tmp.innerHTML = html;
    return (tmp.textContent || tmp.innerText || '').trim();
  }

  function trimWords(text, maxWords) {
    const clean = (text || '').replace(/\s+/g, ' ').trim();
    if (!clean) return '';
    const words = clean.split(' ');
    if (words.length <= maxWords) return clean;
    return words.slice(0, maxWords).join(' ') + '…';
  }

  function excerptFromContent(contentText) {
    // "2 خط اول" را به‌صورت خلاصه‌ای از ابتدای متن می‌سازیم و با CSS روی 2 خط clamp می‌کنیم.
    // این متن مستقیماً از content می‌آید تا با تغییر محتوای پست هم به‌روز شود.
    const clean = (contentText || '').replace(/\s+/g, ' ').trim();
    if (!clean) return '';
    return trimWords(clean, 28);
  }

  function estimateReadingTimeMinutes(text) {
    const words = (text || '').trim().split(/\s+/).filter(Boolean).length;
    const wpm = 200;
    return Math.max(1, Math.round(words / wpm));
  }

  function formatPersianDate(dateString) {
    try {
      const d = new Date(dateString);
      return new Intl.DateTimeFormat('fa-IR', { year: 'numeric', month: 'long', day: 'numeric' }).format(d);
    } catch (e) {
      return '';
    }
  }

  function shuffleArray(arr) {
    const a = arr.slice();
    for (let i = a.length - 1; i > 0; i--) {
      const j = Math.floor(Math.random() * (i + 1));
      [a[i], a[j]] = [a[j], a[i]];
    }
    return a;
  }

  function buildSlide(post, settings) {
    const link = post.link || '#';

    const title = stripHtml(post.title && post.title.rendered);

    const featured = post._embedded && post._embedded['wp:featuredmedia'] && post._embedded['wp:featuredmedia'][0];
    const imgUrl = featured && featured.source_url ? featured.source_url : '';

    let catName = '';
    if (settings.showBadge && post._embedded && post._embedded['wp:term'] && post._embedded['wp:term'][0] && post._embedded['wp:term'][0][0]) {
      catName = post._embedded['wp:term'][0][0].name || '';
    }

    const dateText = settings.showDate ? formatPersianDate(post.date) : '';

    const contentText = stripHtml(post.content && post.content.rendered);
    const readingMin = settings.showReadingTime ? estimateReadingTimeMinutes(contentText) : 0;
    const contentExcerpt = excerptFromContent(contentText);

    const slide = document.createElement('div');
    slide.className = 'swiper-slide';

    const card = document.createElement('div');
    card.className = 'rnc-card';

    const a = document.createElement('a');
    a.href = link;

    const imgWrap = document.createElement('div');
    imgWrap.className = 'rnc-card-img';

    if (settings.showBadge && catName) {
      const badge = document.createElement('span');
      badge.className = 'rnc-badge';
      badge.textContent = catName;
      imgWrap.appendChild(badge);
    }

    const overlay = document.createElement('span');
    overlay.className = 'rnc-overlay';
    imgWrap.appendChild(overlay);

    if (imgUrl) {
      const img = document.createElement('img');
      img.loading = 'lazy';
      img.decoding = 'async';
      img.src = imgUrl;
      img.alt = title;
      imgWrap.appendChild(img);
    }

    const body = document.createElement('div');
    body.className = 'rnc-card-body';

    const h3 = document.createElement('h3');
    h3.className = 'rnc-title';
    h3.textContent = title;
    body.appendChild(h3);

    const meta = document.createElement('div');
    meta.className = 'rnc-meta';

    const right = document.createElement('span');
    right.textContent = dateText;
    meta.appendChild(right);

    const left = document.createElement('span');
    if (settings.showReadingTime) {
      left.textContent = readingMin + ' دقیقه مطالعه';
    } else {
      left.textContent = '';
    }
    meta.appendChild(left);
    body.appendChild(meta);

    // نمایش خلاصه (۲ خط اول از متن پست) زیر تاریخ/متا
    if (contentExcerpt) {
      const p = document.createElement('p');
      p.className = 'rnc-excerpt';
      p.textContent = contentExcerpt;
      body.appendChild(p);
    }

    if (settings.showReadMore) {
      const rm = document.createElement('span');
      rm.className = 'rnc-read-more';
      rm.textContent = 'ادامه مطلب';
      body.appendChild(rm);
    }

    a.appendChild(imgWrap);
    a.appendChild(body);
    card.appendChild(a);
    slide.appendChild(card);

    return slide;
  }

  async function fetchPosts(settings) {
    const baseUrl = (RNC_GLOBALS && RNC_GLOBALS.restUrl) ? RNC_GLOBALS.restUrl : '/wp-json/wp/v2/posts';
    let url;
    try {
      url = new URL(baseUrl, window.location.origin);
    } catch (e) {
      // Fallback: try relative
      url = new URL('/wp-json/wp/v2/posts', window.location.origin);
    }

    url.searchParams.set('per_page', String(settings.postsCount || 9));
    url.searchParams.set('_embed', '1');
    url.searchParams.set('status', 'publish');

    if (settings.categoryId && Number(settings.categoryId) > 0) {
      url.searchParams.set('categories', String(settings.categoryId));
    }

    // "Popular" uses comment_count (available in REST orderby list).
    if (settings.orderMode === 'popular') {
      url.searchParams.set('orderby', 'comment_count');
      url.searchParams.set('order', 'desc');
    } else {
      url.searchParams.set('orderby', 'date');
      url.searchParams.set('order', 'desc');
    }

    const headers = {};
    if (RNC_GLOBALS && RNC_GLOBALS.restNonce) {
      headers['X-WP-Nonce'] = RNC_GLOBALS.restNonce;
    }

    const res = await fetch(url.toString(), {
      credentials: 'same-origin',
      headers
    });
    if (!res.ok) {
      throw new Error('REST error: ' + res.status);
    }
    const data = await res.json();
    if (!Array.isArray(data)) return [];

    if (settings.orderMode === 'random') {
      return shuffleArray(data);
    }

    return data;
  }

  function initOne(rootEl) {
    if (!rootEl || rootEl.dataset.rncInit === '1') return;
    rootEl.dataset.rncInit = '1';

    let settings = {};
    try {
      settings = JSON.parse(rootEl.getAttribute('data-rnc-settings') || '{}');
    } catch (e) {
      settings = {};
    }

    // دایره‌های صفحه‌بندی را غیرفعال کن
    settings.dots = false;

    const swiperEl = rootEl.querySelector('.rnc-swiper');
    const wrapperEl = rootEl.querySelector('.swiper-wrapper');
    const prevBtn = rootEl.querySelector('.rnc-prev');
    const nextBtn = rootEl.querySelector('.rnc-next');
    const paginationEl = rootEl.querySelector('.rnc-pagination');
    const progressBar = rootEl.querySelector('.rnc-progress-bar');

    if (!swiperEl || !wrapperEl) return;

    // Swiper init first, then optionally replace slides after fetch.
    // اگر Swiper لود نشود، fallback ما با scroll-snap کار می‌کند و فلش‌ها هم اسکرول می‌دهند.
    let swiper = null;
    if (typeof window !== 'undefined' && typeof window.Swiper !== 'undefined') {
      try {
        swiper = new Swiper(swiperEl, {
          slidesPerView: 3,
          spaceBetween: Number(settings.spaceBetween || 16),
          loop: !!settings.loop,
          speed: 500,
          rtl: true,
          navigation: (settings.arrows && prevBtn && nextBtn) ? { prevEl: prevBtn, nextEl: nextBtn } : undefined,
          pagination: (settings.dots && paginationEl) ? { el: paginationEl, clickable: true } : undefined,
          autoplay: settings.autoplay ? { delay: Number(settings.autoplaySpeed || 3500), disableOnInteraction: false } : undefined,
          breakpoints: {
            0: { slidesPerView: 1 },
            768: { slidesPerView: 2 },
            1024: { slidesPerView: 3 }
          }
        });
      } catch (e) {
        console.error('[RNC] Swiper init failed', e);
        swiper = null;
        rootEl.classList.add('rnc-no-swiper');
      }
    } else {
      rootEl.classList.add('rnc-no-swiper');
      console.warn('[RNC] Swiper not found on window. Using scroll-snap fallback.');
    }

    // Fallback nav (scroll-snap): move one "page" left/right.
    if (!swiper && wrapperEl && prevBtn && nextBtn) {
      const scrollByPage = function (dir) {
        const slide = wrapperEl.querySelector('.swiper-slide');
        const slideW = slide ? slide.getBoundingClientRect().width : wrapperEl.getBoundingClientRect().width;
        const gap = Number(settings.spaceBetween || 16);
        const delta = (slideW + gap) * dir;
        wrapperEl.scrollBy({ left: delta, behavior: 'smooth' });
      };
      prevBtn.addEventListener('click', function () { scrollByPage(1); });
      nextBtn.addEventListener('click', function () { scrollByPage(-1); });
    }

    if (swiper && settings.autoplay && settings.showProgress && progressBar) {
      swiper.on('autoplayTimeLeft', function (_s, _time, progress) {
        const pct = Math.max(0, Math.min(1, 1 - progress)) * 100;
        progressBar.style.width = pct.toFixed(2) + '%';
      });
    }

    fetchPosts(settings)
      .then(function (posts) {
        if (Array.isArray(posts) && posts.length) {
          // Replace server-rendered slides only when we have real data.
          wrapperEl.innerHTML = '';
          posts.forEach(function (post) {
            wrapperEl.appendChild(buildSlide(post, settings));
          });

          if (swiper) {
            swiper.update();
            // اطمینان از شروع autoplay بعد از آپدیت اسلایدها
            if (settings.autoplay && swiper.autoplay && typeof swiper.autoplay.start === 'function') {
              swiper.autoplay.start();
            }
          }
        } else if (swiper) {
          swiper.update();
          if (settings.autoplay && swiper.autoplay && typeof swiper.autoplay.start === 'function') {
            swiper.autoplay.start();
          }
        }
      })
      .catch(function (err) {
        console.error('[RNC] Fetch posts failed', err);
        // اگر رندر سمت سرور موجود باشد، همان را نگه می‌داریم.
        // فقط در صورتی پیام خطا نشان بده که واقعاً چیزی برای نمایش نیست.
        const hasSlides = wrapperEl.querySelector('.swiper-slide');
        if (!hasSlides) {
          const slide = document.createElement('div');
          slide.className = 'swiper-slide';
          slide.innerHTML = '<div class="rnc-card"><div class="rnc-card-body"><h3 class="rnc-title">خطا در بارگذاری مطالب</h3><div class="rnc-meta"><span>لطفاً صفحه را رفرش کنید</span><span></span></div></div></div>';
          wrapperEl.appendChild(slide);
        }
        if (swiper) {
          swiper.update();
        }
      });
  }

  function initAll() {
    var roots = document.querySelectorAll('.rnc-root[data-rnc-settings]');
    for (var i = 0; i < roots.length; i++) {
      initOne(roots[i]);
    }
  }

  function runInitAll() {
    initAll();
    // چند بار با تأخیر کم تا ویجت‌هایی که Elementor بعداً به DOM اضافه می‌کند هم همگی سریع مقداردهی شوند
    if (typeof requestAnimationFrame !== 'undefined') {
      requestAnimationFrame(initAll);
    }
    setTimeout(initAll, 50);
    setTimeout(initAll, 200);
  }

  // همه کاروسل‌ها را یک‌جا و موازی مقداردهی کن (نه یکی‌یکی با تاخیر Elementor)
  if (typeof document !== 'undefined') {
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', runInitAll);
    } else {
      runInitAll();
    }
  }

  // Elementor: فقط برای ویجت‌هایی که بعداً لود می‌شوند (مثلاً در تب/آکاردئون). قبلاً init شده با rncInit جلوگیری می‌شود.
  if (typeof window !== 'undefined' && typeof elementorFrontend !== 'undefined' && elementorFrontend.hooks) {
    elementorFrontend.hooks.addAction('frontend/element_ready/rnc_news_carousel_cards.default', function ($scope) {
      var el = $scope && ($scope[0] !== undefined) ? $scope[0] : $scope;
      if (el) initOne(el);
    });
  }
})();



