(function () {
	'use strict';

	function RbnTicker(root) {
		this.root = root;
		this.track = root.querySelector('.rbn-track');
		this.items = Array.prototype.slice.call(root.querySelectorAll('.rbn-item'));
		this.prevBtn = root.querySelector('.rbn-prev');
		this.nextBtn = root.querySelector('.rbn-next');
		this.progressBar = root.querySelector('.rbn-progress-bar');

		if (this.items.length === 0 || !this.track) {
			return;
		}

		var data = {};
		try {
			data = JSON.parse(root.getAttribute('data-rbn-settings') || '{}');
		} catch (e) {
			data = {};
		}

		this.autoplay = !!data.autoplay;
		this.autoplaySpeed = data.autoplaySpeed || 3500;
		this.mode = data.scrollMode === 'marquee' ? 'marquee' : 'slide';
		this.pauseOnHover = !!data.pauseOnHover;
		this.showProgress = !!data.showProgress;
		this.showArrows = !!data.showArrows;

		this.currentIndex = 0;
		this.timer = null;
		this.isPaused = false;
		this.marqueeOffset = 0;

		this.setup();
	}

	RbnTicker.prototype.setup = function () {
		var root = this.root;
		var self = this;

		root.dataset.rbnLoading = '0';

		if (this.showArrows && this.prevBtn && this.nextBtn) {
			this.prevBtn.addEventListener('click', function () {
				self.prev();
			});
			this.nextBtn.addEventListener('click', function () {
				self.next();
			});
		}

		if (this.pauseOnHover) {
			root.addEventListener('mouseenter', this.pause.bind(this));
			root.addEventListener('mouseleave', this.resume.bind(this));
		}

		if (this.autoplay) {
			if (this.mode === 'marquee') {
				this.startMarquee();
			} else {
				this.startSlideLoop();
			}
		}
	};

	RbnTicker.prototype.slideTo = function (index) {
		if (this.mode !== 'slide' && this.mode !== 'marquee') {
			return;
		}
		this.currentIndex = (index + this.items.length) % this.items.length;
		var target = this.items[this.currentIndex];
		if (!target) return;

		var offset = target.offsetLeft;
		this.track.style.transform = 'translateX(' + offset * -1 + 'px)';
	};

	RbnTicker.prototype.next = function () {
		if (this.mode === 'slide') {
			this.slideTo(this.currentIndex + 1);
			this.restartSlideLoop();
		}
	};

	RbnTicker.prototype.prev = function () {
		if (this.mode === 'slide') {
			this.slideTo(this.currentIndex - 1);
			this.restartSlideLoop();
		}
	};

	RbnTicker.prototype.startSlideLoop = function () {
		var self = this;
		if (!this.autoplay) return;
		clearInterval(this.timer);

		if (this.showProgress && this.progressBar) {
			this.progressBar.style.transition = 'none';
			this.progressBar.style.width = '0%';
			requestAnimationFrame(function () {
				self.progressBar.style.transition = 'width ' + self.autoplaySpeed + 'ms linear';
				self.progressBar.style.width = '100%';
			});
		}

		this.timer = setInterval(function () {
			if (!self.isPaused) {
				self.slideTo(self.currentIndex + 1);
				if (self.showProgress && self.progressBar) {
					self.progressBar.style.transition = 'none';
					self.progressBar.style.width = '0%';
					requestAnimationFrame(function () {
						self.progressBar.style.transition = 'width ' + self.autoplaySpeed + 'ms linear';
						self.progressBar.style.width = '100%';
					});
				}
			}
		}, this.autoplaySpeed);
	};

	RbnTicker.prototype.restartSlideLoop = function () {
		if (!this.autoplay || this.mode !== 'slide') return;
		this.startSlideLoop();
	};

	RbnTicker.prototype.startMarquee = function () {
		var self = this;
		if (!this.autoplay) return;
		clearInterval(this.timer);

		this.timer = setInterval(function () {
			if (!self.isPaused) {
				self.slideTo(self.currentIndex + 1);
			}
		}, this.autoplaySpeed);
	};

	RbnTicker.prototype.pause = function () {
		this.isPaused = true;
	};

	RbnTicker.prototype.resume = function () {
		this.isPaused = false;
	};

	function initOne(root) {
		if (!root || root.dataset.rbnInit === '1') return;
		root.dataset.rbnInit = '1';
		root.dataset.rbnLoading = '0';
		new RbnTicker(root);
	}

	function initAll() {
		document.querySelectorAll('.rbn-root').forEach(initOne);
	}

	if (typeof window !== 'undefined' && window.elementorFrontend && elementorFrontend.hooks) {
		elementorFrontend.hooks.addAction('frontend/element_ready/rbn_breaking_news_ticker_pro.default', function ($scope) {
			initOne($scope[0] || $scope);
		});
	} else {
		document.addEventListener('DOMContentLoaded', initAll);
	}
})();


