/**
 * Coming Soon Widget – Countdown & form handler
 * No external libraries, vanilla JS.
 */
(function () {
  'use strict';

  function parseDateTime(datetimeStr, tz) {
    var m = datetimeStr.match(/^(\d{4})-(\d{2})-(\d{2})\s+(\d{2}):(\d{2})/);
    if (!m) return null;
    var d = new Date(
      parseInt(m[1], 10),
      parseInt(m[2], 10) - 1,
      parseInt(m[3], 10),
      parseInt(m[4], 10),
      parseInt(m[5], 10),
      0,
      0
    );
    return d.getTime();
  }

  function pad(n) {
    return n < 10 ? '0' + n : String(n);
  }

  function runCountdown(el) {
    var data = el.getAttribute('data-countdown');
    if (!data) return;

    var config;
    try {
      config = JSON.parse(data);
    } catch (e) {
      return;
    }

    var targetMs = parseDateTime(config.datetime, config.tz);
    if (!targetMs) return;

    var container = el.querySelector('.dsoon-countdown');
    var expiryEl = el.querySelector('.dsoon-expiry-message');
    if (!container) return;

    function tick() {
      var now = Date.now();
      var diff = targetMs - now;

      if (diff <= 0) {
        if (config.expiry === 'redirect' && config.redirect) {
          window.location.href = config.redirect;
          return;
        }
        if (config.expiry === 'message' && expiryEl) {
          container.style.display = 'none';
          expiryEl.textContent = config.message || '';
          expiryEl.style.display = 'block';
        } else if (config.expiry === 'hide') {
          container.style.display = 'none';
        }
        return;
      }

      var d = Math.floor(diff / 86400000);
      var h = Math.floor((diff % 86400000) / 3600000);
      var m = Math.floor((diff % 3600000) / 60000);
      var s = Math.floor((diff % 60000) / 1000);

      var dayEl = container.querySelector('[data-unit="days"]');
      var hourEl = container.querySelector('[data-unit="hours"]');
      var minEl = container.querySelector('[data-unit="minutes"]');
      var secEl = container.querySelector('[data-unit="seconds"]');

      if (dayEl) dayEl.textContent = pad(d);
      if (hourEl) hourEl.textContent = pad(h);
      if (minEl) minEl.textContent = pad(m);
      if (secEl) secEl.textContent = pad(s);
    }

    tick();
    var interval = setInterval(tick, 1000);
  }

  function handleForm(el) {
    var form = el.querySelector('.dsoon-form');
    var feedback = el.querySelector('.dsoon-form-feedback');
    if (!form || !feedback) return;

    form.addEventListener('submit', function (e) {
      e.preventDefault();
      var input = form.querySelector('input[type="email"]');
      if (!input) return;

      var email = (input.value || '').trim();
      var successMsg = form.getAttribute('data-success') || 'ثبت شد.';
      var errorMsg = form.getAttribute('data-error') || 'ایمیل معتبر وارد کنید.';

      feedback.style.display = 'block';

      if (!email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
        feedback.textContent = errorMsg;
        feedback.style.color = '#c0392b';
        return;
      }

      feedback.textContent = successMsg;
      feedback.style.color = '#27ae60';
      input.value = '';
    });
  }

  function init() {
    var widgets = document.querySelectorAll('.dsoon-wrapper[data-countdown]');
    widgets.forEach(function (el) {
      runCountdown(el);
      handleForm(el);
    });
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

  if (typeof elementorFrontend !== 'undefined') {
    elementorFrontend.hooks.addAction('frontend/element_ready/disto_coming_soon.default', function ($scope) {
      var el = $scope[0];
      if (el && el.querySelector) {
        runCountdown(el);
        handleForm(el);
      }
    });
  }
})();
